package org.eyelanguage.rl.reading;

import java.util.*;
import java.io.*;
import net.pakl.rl.*;

//import edu.emory.mathcs.backport.java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.ArrayBlockingQueue;
/**
 * Describes how states evolve in response to actions in a serial attention and word identification model.
 */
public class SentenceWorld implements World, Serializable
{
    static final long serialVersionUID = -4499055574151223530L;
    
    RandomPerThread randomSourcePerThread = new RandomPerThread();
    /** Beyond this distance, visual acuity goes non-linearly to zero. Attend actions do not advance attend time
     * at this stage.  Somehow, this has to be penalized, but do we want to avoid sending predicted reward
     * to infinity? */
    public int ACUITY_LIMIT = 1000;
    
    
    /** By how much the 'attended time' dimension is incremented on each simulated timestep. */
    public int ATTEND_INTERVAL = 5;
    
    /** If true, means that as word identifcation time increases linearly as fovea is distanced from
     *  attended word center; otherwise just a constant penalty for any distance from the word center */
    public boolean ACUITY_LINEAR_PENALIZE = true;

    /** Amount of extra time required per character eccentricity. */
    public int ACUITY_LINEAR_PENALTY_SLOPE = ATTEND_INTERVAL;

    /** By how much the 'saccade program time' dimension is incremented on each simualted timestep. */
    public int SACCADE_PROGRAM_INTERVAL = 5;
    
    /** Maximum amount of time to identify a word, ever (is this actually used?) */
    public int MAX_POSSIBLE_ID_TIME = 50;        // ms, max time to identify a word, ever.
    
    /** The actual amount of time it takes to program a saccade. */
    public int MAX_SACCADE_PROG_TIME = 15;       // ms, time it takes to program a saccade.
    
    /** When sentence world is initialized, this number reflects the maximum possible penalty
     * on word identifiation time (which is useful for knowing the bounds of the state space). */
    public int MAX_POSSIBLE_ACUITY_PENALTY = 0;
    /** Maximum amount of time that could possibly be spent identifying a particular word
     *  given that we're looking at the center of the word. */
    public int MAX_POSSIBLE_ATTEND_TIME = MAX_POSSIBLE_ID_TIME + MAX_SACCADE_PROG_TIME;
    
    
    public int MAX_POSSIBLE_PENALTY = 0;
    // milliseconds maximum amount of time it would be possible to attend a word before going into a seizure
    
    /** A larger value here means more error if saccadic error is enabled. */
    public double SACCADIC_ERROR_SPREAD = 2;
    
    /** The amount of time subtracted from attending a word after making a saccade. */
    public int SACCADE_TIME_PENALTY = -5;      // milliseconds of time to make a saccadic eye movement
    
    public int MIN_ATTEND_TIME_WITH_PENALTIES;
    public int MAX_ATTEND_TIME_WITH_PENALTIES;
    
    /** Whether saccadic oculomotor error is enabled. */
    public boolean SACCADIC_ERROR = false;
    
    protected int length;                             // num characters
    boolean isBuilt = false;
    protected String sentenceString = "";
    protected String name = "namelessWorld";
    ArrayList <Word> words = new ArrayList<Word>();              // word items themselves
    
    public boolean CACHE_ENABLED = false;
    public boolean SACCADIC_ERROR_GAUSSIAN = true;

    
    public ReadingStateParallelRelative setStartingState = null;
    
    ReadingStateFactory readingStateFactory = null;
    
    int numStates;
    
    
    protected int minOf(int a, int b)
    {
        if (a < b) return a;
        return b;
    }
    protected int maxOf(int a, int b)
    {
        if (a > b) return a;
        return b;
    }

    int bound_maxDistanceFromAttendCenter = -Integer.MAX_VALUE;
    int bound_minDistanceFromAttendCenter = Integer.MAX_VALUE;

//    public boolean isOutOfBoundsState(State state)
//    {
//        ReadingState rs = (ReadingState) state;
//        if (rs.getDistanceFromAttendCenter() > bound_maxDistanceFromAttendCenter)
//        {
//            return true;
//        }
//        if (rs.getDistanceFromAttendCenter() < bound_minDistanceFromAttendCenter)
//        {
//            return true;
//        }
//        return false;
//    }
        
    public void growBounds(State state)
    {
        ReadingState rs = (ReadingState) state;
        if (rs.getDistanceFromAttendCenter() > bound_maxDistanceFromAttendCenter)
        {
            bound_maxDistanceFromAttendCenter = rs.getDistanceFromAttendCenter();
        }
        if (rs.getDistanceFromAttendCenter() < bound_minDistanceFromAttendCenter)
        {
            bound_minDistanceFromAttendCenter = rs.getDistanceFromAttendCenter();
        }
    }
    
    public static void main(String args[])
    {
        SentenceWorld w = new SentenceWorld("test", 1);
        w.setSentence("123456789", "0.1 0.1", "20 20");
        ReadingState s1 = new ReadingState();
        ReadingState s2 = new ReadingState();
        ReadingState s3 = new ReadingState();
        s1.setTimeAttending(10);
        s2.setTimeAttending(10);
        s3.setTimeAttending(10);
        s1.setAttendedWordLen(9);
        s2.setAttendedWordLen(9);
        s3.setAttendedWordLen(9);
        s1.setDistanceFromAttendCenter(5);
        s3.setDistanceFromAttendCenter(5);
        s2.setDistanceFromAttendCenter(0);
        System.out.println("s1: " + s1);
        System.out.println("s2: " + s2);
        System.out.println("s3: " + s3);
        System.out.println("Adjustment from s1 to s2: " + w.acuityAdjustment(s1, s2));
        System.out.println("Adjustment from s2 to s3: " + w.acuityAdjustment(s2, s3));
        System.out.println("Adjustment from s1 to s3: " + w.acuityAdjustment(s1, s3));
        System.out.println("Adjustment from s3 to s2: " + w.acuityAdjustment(s3, s2));
        System.out.println("Adjustment from s2 to s1: " + w.acuityAdjustment(s2, s1));
    }
    
    /** If we saccade to a position, the time attended is worth less than future time that will be spent (see detail). 
     * In detail it is important to note that the time already spent is taken from newState -- this is ok in the serial
     * attention case because attendWord has not yet been called.  The subtlety becomes more relevant in the parallel
     * attention case, because attendWord *has* already been called by super() [nonparallel sentence world code is 
     * called.]  It seems for parallel processing correctness, if parallel lexical processing is called first, then the
     * acuity adjustment code should subtract some time.*/
    public int acuityAdjustment(ReadingState previousState, ReadingState newState)
    {
        Word attendedWord = getWord(previousState.getAttendedWordID());
        double foveatedCenterIDTime = attendedWord.getIdentificationTime();
        double preSaccadeIDTime  = (double) totalIdentificationTimePlusAcuityEffect(attendedWord, previousState.getEyePosition());
        double postSaccadeIDTime = (double) totalIdentificationTimePlusAcuityEffect(attendedWord, newState.getEyePosition());
        double timeAlreadySpent = (double) newState.getTimeAttending();
        
        double preSaccadeTimeWasWorth = (foveatedCenterIDTime / preSaccadeIDTime) * timeAlreadySpent;
        double postSaccadeTimeIsWorth = (postSaccadeIDTime / foveatedCenterIDTime) * preSaccadeTimeWasWorth;
        
        // Round to nearest this.ATTEND_INTERVAL, rounding down to prevent faster-than-possible identification.
        postSaccadeTimeIsWorth = postSaccadeTimeIsWorth / (double) ATTEND_INTERVAL;
        postSaccadeTimeIsWorth = Math.floor(postSaccadeTimeIsWorth);
        postSaccadeTimeIsWorth = postSaccadeTimeIsWorth * (double) ATTEND_INTERVAL;
        
        int adjustBy = -1 * (int) (timeAlreadySpent - postSaccadeTimeIsWorth);
        return adjustBy;
    }
    
    public int calculateDistFromAttendedWordCenter(ReadingState state)
    {
        Word attendedWord = getWord(state.getAttendedWordID());
        List eyePositions = this.getPossibleEyePositions(attendedWord);
        int center = ((Integer) eyePositions.get((eyePositions.size()-1) / 2)).intValue();
        return state.getEyePosition() - center;
    }
    
    protected int totalIdentificationTimePlusAcuityEffect(int wid, int fromEyePosition)
    {
        return totalIdentificationTimePlusAcuityEffect(getWord(wid), fromEyePosition);
    }
    
    protected int totalIdentificationTimePlusAcuityEffect(Word w, int fromEyePosition)
    {
        int result = w.getIdentificationTime() - acuityIdentificationPenalty(w,fromEyePosition);
        return result;
    }

    public int getCenterOfWord(Word w)
    {
        int centerOfW = -1;
        for (int i = 0; i < w.getID(); i++)
        {
            centerOfW = 1 + centerOfW + this.getWord(i).getLength();
        }
        return 1 + centerOfW + (int)  ((double) w.getLength() / 2.0d);
    }
    public int getCenterOfWord(int wid)
    {
        return getCenterOfWord(getWord(wid));
    }
    
    String requester;
    public void setRequester(String r)
    {
        synchronized(this)
        {
            requester = r;
        }
    }
    
    public State getNewState(State oldGenericState, Action genericAction, String r)
    {
        synchronized(this)
        {
            requester = r;
            State result = getNewState(oldGenericState, genericAction);
            requester = "requester";
            return result;
        }
    }
    
    /**
     * This critical method returns the new state given an action
     * from an old state, and in this case, simply adds state to
     * position to simulate movement.  <B>Note that there are no boundary
     * conditions set up here.</B>
     *
     *  ROADMAP
     *  -------
     *
     *  handle action: executing saccade
     *  else
     *  not executing saccade (staying in current eye position)
     *
     *  handle action: requesting saccade (start programming)
     *
     *  handle action: attend next word
     *
     *  handle state: programming saccade
     *
     */
    public State getNewState(State oldGenericState, Action genericAction)
    {
        ReadingState oldState = (ReadingState) oldGenericState;
        ReadingAction readingAction = (ReadingAction) genericAction;
        if (readingAction.isRemainHere())
        {
            ReadingState newState = readingStateFactory.createNewReadingState(oldState);
            return (State) newState;
        }
        
        if (this.isTerminalState(oldState))
        {
            ReadingState newState = readingStateFactory.createNewReadingState(oldState);
            return (State) newState;
        }
        
        ReadingState newState = readingStateFactory.createNewReadingState(oldState);
        
        
        if (readingAction.isAttendNextWord())
        {
            try
            {
                attemptAttendNextWord(newState);
            }
            catch (RuntimeException e)
            {
                e.printStackTrace();
                System.err.println("Action was " + readingAction);
                throw e;
            }
        }
        
        if (newState.isProgrammingSaccade())
        {
            if (newState.getTimeInProgrammingSaccade() >= this.MAX_SACCADE_PROG_TIME)
            {
                newState = executeSaccade(newState);
            }
            else
            {
                newState.setTimeInProgrammingSaccade(SACCADE_PROGRAM_INTERVAL + newState.getTimeInProgrammingSaccade());
            }
        }
        
        if (readingAction.isRequestingSaccade())        // Initiate saccade programming
        {
            newState.setProgrammingSaccade(true);
            newState.setTimeInProgrammingSaccade(0);
            newState.setSaccadeRequestDistance(readingAction.getSaccadeRequestDistance());
        }
        
        
        newState = checkForAcuityLimit(newState);  // Must be done before possibly acuityAdjustments
        
        
        if (oldState.isProgrammingSaccade() && !newState.isProgrammingSaccade())
        {
            newState.setTimeAttending(newState.getTimeAttending() + acuityAdjustment(oldState, newState));
            
            if (!readingAction.isAttendNextWord())  // It's ok to add Saccade Penalty Timestep only if not shifting attention. [Don't add it twice when att shifts.]
            {
                newState.setTimeAttending(SACCADE_TIME_PENALTY + newState.getTimeAttending());
            }
            
            if (newState.getTimeAttending() < MIN_ATTEND_TIME_WITH_PENALTIES)
            {
                newState.setTimeAttending(MIN_ATTEND_TIME_WITH_PENALTIES);
            }
            if (newState.getTimeAttending() >= MAX_ATTEND_TIME_WITH_PENALTIES)
            {
                newState.setTimeAttending(MAX_ATTEND_TIME_WITH_PENALTIES-ATTEND_INTERVAL);
            }
        }
        
        // Apply attention last of all to prevent messing up acuity calculations.
        // The agent can shift attention, or eyes, and immediately attend the next word.
        if (readingAction.isAttendThisWord())
        {
            newState = attendCurrentWord(newState);
        }
        
        newState = checkForWordIdentification(newState);
        
        
        return newState;
        
    }

    /** If the eyes land too eccentricly, force eyes to land within acuity limit. */
    public ReadingState checkForAcuityLimit(ReadingState newState)
    {
        // Overshoot
        if (newState.getDistanceFromAttendCenter() > this.ACUITY_LIMIT)
        {
            int overshoot = newState.getDistanceFromAttendCenter() - ACUITY_LIMIT;
            newState.setEyePosition(newState.getEyePosition() - overshoot);
            newState.setFixatedWordID(this.getWordAtEyePosition(newState.getEyePosition()).getID());
            newState.setDistanceFromAttendCenter(newState.getDistanceFromAttendCenter() - overshoot);
        }
        // Undershoot
        if (newState.getDistanceFromAttendCenter() < -this.ACUITY_LIMIT)
        {
            int undershoot = Math.abs(Math.abs(newState.getDistanceFromAttendCenter()) - ACUITY_LIMIT);
            newState.setEyePosition(newState.getEyePosition() + undershoot);
            newState.setFixatedWordID(this.getWordAtEyePosition(newState.getEyePosition()).getID());
            newState.setDistanceFromAttendCenter(newState.getDistanceFromAttendCenter() + undershoot);
        }
        
        return newState; 
    }
    
    public int applySaccadicError(int saccadeRequestDistance)
    {
        long result = 0;
        
        double zForHalf = 0.3829;
        double zForOneAndHalf = 0.8664;
        
        Random random = (Random) randomSourcePerThread.get();
        double x = random.nextDouble();
        // double x = randomNumberGenerator.nextDouble();
        if (SACCADIC_ERROR_GAUSSIAN)
        {
            if (x <= zForHalf) result = saccadeRequestDistance;
            else
                if (x <= (zForHalf + 0.5d*(zForOneAndHalf-zForHalf))) result = saccadeRequestDistance-1;
                else
                    if (x <= (zForHalf + 1.0d*(zForOneAndHalf-zForHalf))) result = saccadeRequestDistance+1;
                    else
                        if (x <= (zForOneAndHalf + 0.5d*(1.00d-zForOneAndHalf))) result = saccadeRequestDistance-2;
                        else
                            if (x <= (zForOneAndHalf + 1.0d*(1.00d-zForOneAndHalf))) result = saccadeRequestDistance+2;
        }
        else
        {
            // Uniform error
            x = x - 0.5d;       // centered
            x = x * 2.0d * (SACCADIC_ERROR_SPREAD + 0.5d);         // spread
            result = Math.round(saccadeRequestDistance + x);
        }
        return (int) result;
    }
    
    protected ReadingState executeSaccade(ReadingState oldState)
    {
        ReadingState newState = readingStateFactory.createNewReadingState();
        newState.copyFrom(oldState);
        newState.setProgrammingSaccade(false);
        newState.setTimeInProgrammingSaccade(0);
        int newEyePosition = 0;
        
        if (SACCADIC_ERROR == false)
        {
            newEyePosition = newState.getSaccadeRequestDistance() + newState.getEyePosition();
        }
        else
        {
            newEyePosition = applySaccadicError(newState.getSaccadeRequestDistance()) + oldState.getEyePosition();
        }
        
        if (newEyePosition < 0)
        {
            newEyePosition = 0;
        }
        if (newEyePosition >= this.getSentenceLength())
        {
            newEyePosition = this.getSentenceLength()-1;
        }
        
        newState.setEyePosition(newEyePosition);
        newState.setFixatedWordID(this.getWordAtEyePosition(newState.getEyePosition()).getID());
        newState.setSaccadeRequestDistance(0);
        newState.setDistanceFromAttendCenter(this.calculateDistFromAttendedWordCenter(newState));
        return newState;
    }
    
    public ReadingState checkForWordIdentification(ReadingState newState)
    {
        int attWordID = newState.getAttendedWordID();
        Word currentWord = getWord(attWordID);
        int wordIdTime = totalIdentificationTimePlusAcuityEffect(currentWord, newState.getEyePosition());
        int actualWordIdTime = minOf(this.MAX_ATTEND_TIME_WITH_PENALTIES, wordIdTime);
        if (newState.getTimeAttending() >= actualWordIdTime)
        {
            newState.setIdentified(true);
        }
        else
        {
            newState.setIdentified(false);
        }
        return newState;
    }
    
    public ReadingState attendCurrentWord(ReadingState newState)
    {
        // Always automatically advance in state space in attended time dimension
        newState.setTimeAttending(this.ATTEND_INTERVAL + newState.getTimeAttending());
        newState.setDistanceFromAttendCenter(this.calculateDistFromAttendedWordCenter(newState));
        
        int attWordID = newState.getAttendedWordID();
        Word currentWord = getWord(attWordID);
        newState.setAttendedWordLen(currentWord.getPsychologicalLength());
        newState.setAttendedWordPredictability(currentWord.getPredictabilityFromPreceedingWord());
        
        newState.setPreviousWordLen(getWordLengthOrZero(attWordID - 1));
        newState.setNextWordLen     (getWordLengthOrZero(attWordID + 1));
        newState.setNextNextWordLen(getWordLengthOrZero(attWordID + 2));
        
        newState.setPreviousWordPredictability  (getWordPredictabilityOrZero(attWordID - 1));
        newState.setNextWordPredictability      (getWordPredictabilityOrZero(attWordID + 1));
        newState.setNextNextWordPredictability  (getWordPredictabilityOrZero(attWordID + 2));
        
        return newState;
    }
    
    protected int getWordLengthOrZero(int id)
    {
        if ((id < 0) || (id >= this.getNumWords())) return 0;
        return getWord(id).getPsychologicalLength();
    }
    protected double getWordPredictabilityOrZero(int id)
    {
        return 0;
    }
    
    
    protected void attemptAttendNextWord(ReadingState newState)
    {
        if ((1 + newState.getAttendedWordID()) >= this.getNumWords())
        {
            throw new RuntimeException("Error: Tried to attend word beyond end of sentence from a non-terminal state.  Shouldn't there have been a REMAIN action from that state?\n Original State was "+newState);
        }
        Word currentWord = getWord(1 + newState.getAttendedWordID());
        
        newState.setAttendedWordID(currentWord.getID());
        
        newState.setAttendedWordLen(currentWord.getPsychologicalLength());
        newState.setAttendedWordPredictability(currentWord.getPredictabilityFromPreceedingWord());
        
        newState.setPreviousWordLen(getWordLengthOrZero(currentWord.getID() - 1));
        newState.setNextWordLen     (getWordLengthOrZero(currentWord.getID() + 1));
        newState.setNextNextWordLen(getWordLengthOrZero(currentWord.getID() + 2));
        
        newState.setPreviousWordPredictability  (getWordPredictabilityOrZero(currentWord.getID() - 1));
        newState.setNextWordPredictability      (getWordPredictabilityOrZero(currentWord.getID() + 1));
        newState.setNextNextWordPredictability  (getWordPredictabilityOrZero(currentWord.getID() + 2));
        
        newState.setTimeAttending(0);       // We have just started attending this word, AND WE DID NOT APPLY ATTENTION DURING THE
        // TIMESTEP WHEN ATTENTION MOVED. (THAT'S WHY IT'S 0 INSTEAD OF ATTEND_INTERVAL.)
        newState.setIdentified(false);
        
        newState.setDistanceFromAttendCenter(this.calculateDistFromAttendedWordCenter(newState));
    }
    
    
    public ReadingState getNewState(ReadingState rs, ReadingAction ra)
    {
        return (ReadingState) getNewState((State) rs, (Action) ra);
    }
    
    
    public int distance(State state1, State state2)
    {
        ReadingState rs1 = (ReadingState) state1;
        ReadingState rs2 = (ReadingState) state2;
        return Math.abs(rs2.getEyePosition() - rs1.getEyePosition());
    }
    
    public int getSentenceLength()
    {
        return this.sentenceString.length();
    }
    
    public void setSentence(String newSentenceString, String predictabilities, String minIDTimes)
    {
        setSentence(newSentenceString, predictabilities, minIDTimes, "", false);
    }

    public void setSentence(String newSentenceString, String predictabilities, String minIDTimes, String lengths, boolean lengthsManuallyOverridden)
    {
        this.sentenceString = newSentenceString;
        if (this.sentenceString.charAt(this.sentenceString.length()-1) != ' ')
        {
            System.out.println("Note: Appending space character at end of sentence automatically.");
            this.sentenceString = this.sentenceString + " ";
        }
        System.out.println("Sentence Length observed as "+this.getSentenceLength());
        Word preceedingWord = null;
        int wordCounter = 0;
        words = new ArrayList();
        StringTokenizer wordTokenizer = new StringTokenizer(sentenceString, " ", false);
        StringTokenizer predTokenizer = new StringTokenizer(predictabilities, " ", false);
        StringTokenizer timeTokenizer = new StringTokenizer(minIDTimes, " ", false);
        StringTokenizer lengthTokenizer = new StringTokenizer(lengths, " ", false);
        while (wordTokenizer.hasMoreTokens())
        {
            Word thisWord = new Word();
            thisWord.setID(wordCounter);
            thisWord.setText(wordTokenizer.nextToken());
            try
            {
                thisWord.setIdentificationTime(new Integer(timeTokenizer.nextToken()).intValue());
            }
            catch (NoSuchElementException nse)
            {
                throw new NoSuchElementException("While processing word "+wordCounter+" could not find id time.\n"+sentenceString+"\n"+predictabilities+"\n"+minIDTimes);
            }
            if (lengthsManuallyOverridden) thisWord.setLengthOverride(new Integer(lengthTokenizer.nextToken()).intValue());
            
            thisWord.setPredictabilityFromPreceedingWord(new Double(predTokenizer.nextToken()));
            thisWord.setPreceedingWord(preceedingWord);
            if (thisWord.getIdentificationTime() > this.MAX_POSSIBLE_ID_TIME) this.MAX_POSSIBLE_ID_TIME = thisWord.getIdentificationTime();
            words.add(thisWord);
            preceedingWord = thisWord;
            wordCounter++;
        }
        // The following is a negative number
        this.MAX_POSSIBLE_ATTEND_TIME = MAX_POSSIBLE_ID_TIME + MAX_SACCADE_PROG_TIME;
        this.MAX_POSSIBLE_ACUITY_PENALTY = this.acuityIdentificationPenaltyAbsolute(Math.abs(sentenceString.length()));
        
        System.out.println("ACUITY_LINEAR_PENALIZE = " + this.ACUITY_LINEAR_PENALIZE);
        System.out.println("Sentence length is " + sentenceString.length() + " so max acuity penalty is " + MAX_POSSIBLE_ACUITY_PENALTY);
        
        this.MAX_POSSIBLE_PENALTY = 0 + this.SACCADE_TIME_PENALTY + MAX_POSSIBLE_ACUITY_PENALTY;
        
        this.MIN_ATTEND_TIME_WITH_PENALTIES = 0 + MAX_POSSIBLE_PENALTY;
        this.MAX_ATTEND_TIME_WITH_PENALTIES = MAX_POSSIBLE_ATTEND_TIME + Math.abs(MAX_POSSIBLE_PENALTY);
        System.out.println(getSettings());
        if (isBuilt) build();               // rebuild if necessary
    }
    
        
    public int acuityIdentificationPenalty(int wid, int currentEyePosition)
    {
        return acuityIdentificationPenalty(getWord(wid), currentEyePosition);
    }

    public int acuityIdentificationPenalty(Word w, int currentEyePosition)
    {
        int centerOfW = getCenterOfWord(w);
        int distanceFromCenterOfW = Math.abs(currentEyePosition - centerOfW);
        
        return acuityIdentificationPenaltyAbsolute(distanceFromCenterOfW);
    }
    
    public int acuityIdentificationPenaltyAbsolute(int absoluteDistanceInCharacters)
    {
        if (ACUITY_LINEAR_PENALIZE == true)
        {
            return -1 * ACUITY_LINEAR_PENALTY_SLOPE * absoluteDistanceInCharacters;
        }
        else
        {
            // CONSTANT PENALTY IS DEFAULT
            if (absoluteDistanceInCharacters > 0) return -2 * ATTEND_INTERVAL;
            return 0;
        }
    }
    
    public boolean isTerminalState(State state)
    {
        ReadingState readingState = (ReadingState) state;
        if (readingState.isIdentified() && (readingState.getAttendedWordID() == (this.getNumWords()-1)))
        {
            return true;
        }
        return false;
    }
    
    public int getNumberOfPositions()
    {
        return this.numStates;
    }
    
    protected HashSet terminalStatesCache = null;
    
    public HashSet getTerminalStates()
    {
        throw new RuntimeException("No Terminal States list is needed, just inquire using isTerminalState().");
    }
    
    public HashSet getTerminalStates_notUsed()
    {
        if (terminalStatesCache == null)
        {
            System.err.println("SentenceWorld "+this.getName()+": Finding terminal states");
            HashSet result = new HashSet();
            
            int lastWordID = this.getNumWords() - 1;
            
            requester = "terminalStates";
            int visitedCount = 0;
            Iterator i = stateIterator();
            while (i.hasNext())
            {
                visitedCount++;
                ReadingState rs = (ReadingState) i.next();
                if (rs.isIdentified() && (rs.getAttendedWordID() == lastWordID))
                {
                    if (rs instanceof ReadingStateRelative)
                    {
                        ReadingStateRelative r = new ReadingStateRelative((ReadingStateRelative)rs);
                        result.add(r);
                    }
                    else
                        if (rs instanceof ReadingStateParallelRelative)
                        {
                        ReadingStateParallelRelative r = new ReadingStateParallelRelative((ReadingStateParallelRelative)rs);
                        result.add(r);
                        }
                    if (rs instanceof ReadingState)
                    {
                        ReadingState r = new ReadingState((ReadingState)rs);
                        result.add(r);
                    }
                }
            }
            System.err.println("SentenceWorld "+this.getName()+": Finding terminal states. DONE.");
            terminalStatesCache = result;
            return result;
        }
        else
        {
            System.err.println("SentenceWorld "+this.getName()+": Finding terminal states [used cache]");
            return terminalStatesCache;
        }
    }
    
    public List getDeadendStates()
    {
        System.err.println("SentenceWorld "+this.getName()+": Finding dead-end states");
        ArrayList result = new ArrayList();
        Iterator i = stateIterator();
        while (i.hasNext())
        {
            ReadingState rs = (ReadingState) i.next();
            if (rs.getTimeAttending() == this.MAX_ATTEND_TIME_WITH_PENALTIES)
            {
                result.add(rs);
            }
        }
        System.err.println("SentenceWorld "+this.getName()+": Finding dead-end states. DONE.");
        return result;
    }
    
    public void setupReadingStateFactory(int kind, boolean pooling)
    {
        this.readingStateFactory = new ReadingStateFactory(kind, pooling);
    }
    public void setupReadingStateFactory(int kind)
    {
        this.readingStateFactory = new ReadingStateFactory(kind);
    }
    
        
    
    
    /**
     * Here we construct the state space for this sentence.
     *
     */
    public void build()
    {
        // ---------------------------------------------------------------------------------
        // Here we used to build the world, but since the world got so big, here we'll just
        // count up the states.  The world is actually built state-by-state, on the fly,
        // as each state is requested.
        // ---------------------------------------------------------------------------------
        //if (this instanceof SentenceWorldParallel)
        //{
        //    System.out.println("Parallel Sentence World: Not counting up states.");
        //}
        //else
        {
            System.err.println(this.getClass() + ": Counting up states...");
            long timerStart = System.currentTimeMillis();

            Iterator i = stateIterator();
            this.numStates = 0;

            {
                while (i.hasNext())
                {
                    State s = (State) i.next();
                    growBounds(s);
                    this.numStates++;
                    if (numStates % 100000 == 0)
                    { 
                        System.out.println(numStates + " states so far.\n"+s); 
                    }
                    // System.out.println(s);
                }
            }
            long timerElapsed = System.currentTimeMillis() - timerStart;
            System.out.println("SentenceWorld has "+numStates+" states");
            System.out.println("It took " + timerElapsed + " time to count up states, " + ((double) timerElapsed / (double) numStates) + "ms per state");
        }
        isBuilt = true;
    }
    
    
    
    public Iterator stateIterator()
    {
        StateIteratorIndividualist result = new StateIteratorIndividualist();
        return result;
    }
    
    public void setBoundedBufferSize(int n)
    {
        boundedBufferSize = n;
    }
    
    boolean starvedBoolean = false;
    int boundedBufferSize = 10000;
    public class StateIteratorIndividualist extends Thread implements Iterator, Serializable
    {
        private boolean running = true;
        private ArrayBlockingQueue boundedBuffer = null;
        private boolean ceaseRunning = false;
        
        
        public synchronized void emptyBuffer()
        {
            while (boundedBuffer.size() > 0)
            {
                ReadingState rs = (ReadingState) next();
            }
        }
        volatile boolean deliveredLast = false;
        volatile Object nextToReturn = null;
        volatile Object LAST_OBJECT_MARKER = new ReadingState();
        
        public boolean hasNext()
        {
            if (deliveredLast) return false;
            if (nextToReturn == null)
            {
                try
                {
                    nextToReturn = boundedBuffer.take();
                }
                catch (InterruptedException ie)
                {
                    ie.printStackTrace();
                }
            }
            if (nextToReturn == LAST_OBJECT_MARKER)
            {
                deliveredLast = true;
                return false;
            }
            return true;
        }
        
        public StateIteratorIndividualist()
        {
            boundedBuffer = new ArrayBlockingQueue(boundedBufferSize);
            this.setDaemon(true);
            if (readingStateFactory == null)
            {
                readingStateFactory = new ReadingStateFactory();
            }
            this.start();
            // Thread.dumpStack();
        }
        
        public synchronized Object next()
        {
            Object returned = nextToReturn;
            nextToReturn = null;
            return returned;
        }
        
        public void run()
        {
            try
            {
                running = true;
                loadInNext();
                running = false;
                ceaseRunning = false;
            }
            catch (Exception e)
            {
                e.printStackTrace();
                System.exit(1);
            }
        }
        
        
        
        public void loadInNext()
        {
            
            ReadingState tempPreservePrevious = readingStateFactory.createNewReadingState();
            try
            {
                for (int i1 = 0; i1 < words.size(); i1++)
                {
                    Word fixatedWord = (Word) words.get(i1);
                    ArrayList possibleEyePositions = (ArrayList) getPossibleEyePositions(fixatedWord);
                    
                    for (int i2 = 0; i2 < possibleEyePositions.size(); i2++)
                    {
                        Integer eyePosition = (Integer) possibleEyePositions.get(i2);
                        for (int i3 = 0; i3 < words.size(); i3++)
                        {
                            
                            
                            Word attendedWord = (Word) words.get(i3);
                            
                            for (int timeAttended = MIN_ATTEND_TIME_WITH_PENALTIES; timeAttended <= MAX_ATTEND_TIME_WITH_PENALTIES; timeAttended+=ATTEND_INTERVAL)
                            {
                                ReadingState newStateAttending = readingStateFactory.createNewReadingState();
                                
                                newStateAttending.setEyePosition(eyePosition.intValue());
                                newStateAttending.setAttendedWordID(attendedWord.getID());
                                newStateAttending.setAttendedWordLen(attendedWord.getPsychologicalLength());
                                
                                newStateAttending.setAttendedWordPredictability(attendedWord.getPredictabilityFromPreceedingWord());
                                newStateAttending.setPreviousWordLen(getWordLengthOrZero(attendedWord.getID() - 1));
                                newStateAttending.setNextWordLen     (getWordLengthOrZero(attendedWord.getID()+ 1));
                                newStateAttending.setNextNextWordLen(getWordLengthOrZero(attendedWord.getID()+ 2));
                                
                                newStateAttending.setPreviousWordPredictability  (getWordPredictabilityOrZero(attendedWord.getID() - 1));
                                newStateAttending.setNextWordPredictability      (getWordPredictabilityOrZero(attendedWord.getID() + 1));
                                newStateAttending.setNextNextWordPredictability  (getWordPredictabilityOrZero(attendedWord.getID()+ 2));
                                
                                newStateAttending.setFixatedWordID(fixatedWord.getID());
                                newStateAttending.setTimeAttending(timeAttended);
                                newStateAttending.setProgrammingSaccade(false);
                                newStateAttending.setIdentified(false);
                                newStateAttending.setDistanceFromAttendCenter(calculateDistFromAttendedWordCenter(newStateAttending));
                                if (newStateAttending.getTimeAttending() == MAX_ATTEND_TIME_WITH_PENALTIES)
                                {
                                }
                                else
                                    if (Math.abs(newStateAttending.getDistanceFromAttendCenter()) > ACUITY_LIMIT)
                                    {
                                    }
                                    else
                                    {
                                    newStateAttending = checkForWordIdentification(newStateAttending);
                                    boundedBuffer.put(newStateAttending);
                                    }
                                tempPreservePrevious.copyFrom(newStateAttending);
                                
                                if (ceaseRunning)
                                { return; }
                                for (int timeInSaccadicProg = 0; timeInSaccadicProg <= MAX_SACCADE_PROG_TIME; timeInSaccadicProg+=SACCADE_PROGRAM_INTERVAL)
                                {
                                    int maxBack = 10;
                                    int maxFwd = 10;
                                    int actualBack = maxBack;
                                    int actualFwd = maxFwd;
                                    if (eyePosition.intValue() < maxBack)
                                    { 
                                        actualBack = eyePosition.intValue(); 
                                    }
                                    if ((getSentenceLength() - eyePosition.intValue()) <= maxFwd)
                                    { 
                                        actualFwd = getSentenceLength() - eyePosition.intValue(); 
                                    }
                                    
                                    for (int saccRequestDistance = -actualBack; saccRequestDistance <= actualFwd; saccRequestDistance++)
                                    {
                                        if ((saccRequestDistance !=0)
                                        && ((newStateAttending.getTimeAttending()-timeInSaccadicProg) >= MIN_ATTEND_TIME_WITH_PENALTIES))
                                        {
                                            ReadingState newStateProgramming = readingStateFactory.createNewReadingState(tempPreservePrevious);
                                            newStateProgramming.setTimeInProgrammingSaccade(timeInSaccadicProg);
                                            newStateProgramming.setProgrammingSaccade(true);
                                            newStateProgramming.setIdentified(false);
                                            newStateProgramming.setSaccadeRequestDistance(saccRequestDistance);
                                            newStateProgramming.setDistanceFromAttendCenter(calculateDistFromAttendedWordCenter(newStateProgramming));
                                            tempPreservePrevious.copyFrom(newStateProgramming);
                                            if (newStateAttending.getTimeAttending() == MAX_ATTEND_TIME_WITH_PENALTIES)
                                            {
                                                // Do not add any state where non-identified and spent max time
                                                // because we don't want to *start* from that state and fall off the edge
                                                // of state space.
                                            }
                                            else
                                                if (Math.abs(newStateProgramming.getDistanceFromAttendCenter()) > ACUITY_LIMIT)
                                                {
                                                }
                                                else
                                                //if (Math.abs(newStateProgramming.getDistanceFromAttendCenter()+newStateProgramming.getSaccadeRequestDistance()) > ACUITY_LIMIT)
                                                //{
                                                //}
                                                //else
                                                {
                                                newStateProgramming = checkForWordIdentification(newStateProgramming);
                                                boundedBuffer.put(newStateProgramming);
                                                }
                                            if (ceaseRunning)
                                            { return; }
                                        }
                                        
                                    }
                                }
                            }
                        }
                        
                    }
                }
                
                boundedBuffer.put(LAST_OBJECT_MARKER);
                
            }
            catch (InterruptedException ie)
            {
                ie.printStackTrace();
                System.err.println("should not happen.");
                System.exit(1);
            }
        }
        
        public void remove()
        {
        }
    }
    
    
    public List getAllStatesFromEyePosition(int eyePosition)
    {
        ArrayList result = new ArrayList();
        Iterator i = stateIterator();
        while (i.hasNext())
        {
            ReadingState rs = (ReadingState) i.next();
            if (rs.getEyePosition() == eyePosition)
            {
                result.add(rs);
            }
        }
        return result;
    }
    
    public boolean isOnWord(int eyePosition)
    {
        if (this.sentenceString.charAt(eyePosition) == ' ')
        {
            return false;
        }
        return true;
    }
    
    public Word getWordAtEyePosition(int eyePosition)
    {
        for (int i = 0; i < words.size(); i++)
        {
            Word w = (Word) words.get(i);
            ArrayList list = (ArrayList) getPossibleEyePositions(w);
            
            if (getPossibleEyePositions(w).contains(new Integer(eyePosition)))
            {
                // System.out.println("Eye positions of "+w+" are "+getPossibleEyePositions(w));
                return w;
            }
        }
        throw new RuntimeException("Landed on non-claimed space "+eyePosition+" eye position.  Policy/toolbox should have prevented this.");  // throw exception:  Eye cannot land on non-word-space
    }
    public List getAllStatesWhereAttendingWordAt(int eyePosition)
    {
        Word wordAtEyePosition = getWordAtEyePosition(eyePosition);
        
        ArrayList result = new ArrayList();
        Iterator i = stateIterator();
        while (i.hasNext())
        {
            ReadingState rs = (ReadingState) i.next();
            if (rs.getAttendedWordID() == wordAtEyePosition.getID())
            {
                result.add(rs);
            }
        }
        return result;
    }
    
    
    HashMap eyePositionCache = new HashMap();
    public List <Integer> getPossibleEyePositions(Word thisWord)
    {
        if (eyePositionCache.get(thisWord) == null)
        {
            ArrayList result = new ArrayList(thisWord.getLength());
            int start = 0;
            for (int i = 0; i < thisWord.getID(); i++)
            {
                Word w = (Word) words.get(i);
                start = start + w.getLength() + 1;
            }
            int end = start + thisWord.getLength();
            
            if ((end + 1) <= this.sentenceString.length()) end++;       // include final space if one exists
            for (int i = start; i < end; i++)
            {
                result.add(new Integer(i));
            }
            eyePositionCache.put(thisWord, result);
            return (List) result;
        }
        else
        {
            return (List) eyePositionCache.get(thisWord);
        }
    }

    
    
    public List getPossibleEyePositionsSlow(Word thisWord)
    {
        int wordStart = 0;
        
        int previousWordLengthIncludingSpace = 0;
        Iterator wordIterator = words.iterator();
        Word candidateWord = (Word) wordIterator.next();
        
        int wordEnd = wordStart + candidateWord.getLength();
        
        if (thisWord.getID() != candidateWord.getID())
        {
            while (wordIterator.hasNext())
            {
                previousWordLengthIncludingSpace = candidateWord.getLength() + 1;
                candidateWord = (Word) wordIterator.next();
                wordStart = wordStart + previousWordLengthIncludingSpace;
                wordEnd = wordStart + candidateWord.getLength();
                if (candidateWord.getID() == thisWord.getID()) break;
            }
        }
        
        ArrayList result = new ArrayList();
        for (int i = wordStart; i <= wordEnd; i++)
        {
            if (i < sentenceString.length())        // even though space following word is part of word, don't go off edge of sentence
                result.add(new Integer(i));
        }
        return result;
    }
    
    
    public String getSettings()
    {
        String result = "";
        result += "\nATTEND_INTERVAL (steps thru state space time while attending)        : " + ATTEND_INTERVAL;
        result += "\nSACCADE_PROGRAM_INTERVAL (steps thru state space time while prg sacc): " + SACCADE_PROGRAM_INTERVAL;
        result += "\nSACCADE_TIME_PENALTY (ms of time for saccade, penalize attend time)  : " + SACCADE_TIME_PENALTY;
        result += "\nMAX_POSSIBLE_ACUITY_PENALTY = " + MAX_POSSIBLE_ACUITY_PENALTY;
        result += "\nMIN_ATTEND_TIME_WITH_PENALTIES (worst saccades could translate into negative time: "+MIN_ATTEND_TIME_WITH_PENALTIES;
        result += "\nMAX_SACCADE_PROG_TIME (maximum possible program time in saccade prog'ing) : " + MAX_SACCADE_PROG_TIME;
        result += "\nMAX_POSSIBLE_ATTEND_TIME (maximum possible attend time to consder in learning): " + MAX_POSSIBLE_ATTEND_TIME;
        result += "\nMAX_ATTEND_TIME_WITH_PENALTIES (including vis acuity penalities): " + MAX_ATTEND_TIME_WITH_PENALTIES;
        result += "\nATTEND TIME LIMITS RANGE FROM " + (0 + MAX_POSSIBLE_PENALTY) + "ms TO " +(MAX_POSSIBLE_ATTEND_TIME + Math.abs(MAX_POSSIBLE_PENALTY))+"ms";
        result += "\nSACCADIC_ERROR = " + SACCADIC_ERROR;
        result += "\nACUITY_LIMIT = " + ACUITY_LIMIT; 
        result += "\nSACCADIC_ERROR_GAUSSIAN = " + SACCADIC_ERROR_GAUSSIAN;
        
        return result;
    }
    
    public String toText()
    {
        return toString();
    }
    
    public String toString()
    {
        StringBuffer result = new StringBuffer();
        Iterator i = stateIterator();
        while (i.hasNext())
        {
            ReadingState s = (ReadingState) i.next();
            result.append(s.toString());
        }
        return result.toString();
    }
    
    public int getNumWords()
    {
        return words.size();
    }
    
    public Word getWord(int id)
    {
        return (Word) words.get(id);
    }
    
    public boolean isLastWordID(int wordID)
    {
        return wordID == (getNumWords()-1);
    }
    
    
    public SentenceWorld(String newName, long randomSeed)
    {
        this.name = newName;
        randomSourcePerThread.setRandomSeed(randomSeed);
        // this.randomNumberGenerator = new java.util.Random(randomSeed);
    }
    
    public SentenceWorld(String newName)
    {
        this.name = newName;
        randomSourcePerThread.setRandomSeed(1111);
        // this.randomNumberGenerator = new java.util.Random(1111);
    }
    
    public String getName()
    {
        return this.name;
    }
    public boolean getIsBuilt()
    {
        return isBuilt;
    }
    
    public int getATTEND_INTERVAL()
    {
        return ATTEND_INTERVAL;
    }
    
    public void setATTEND_INTERVAL(int ATTEND_INTERVAL)
    {
        this.ATTEND_INTERVAL = ATTEND_INTERVAL;
    }
    
    public int getSACCADE_PROGRAM_INTERVAL()
    {
        return SACCADE_PROGRAM_INTERVAL;
    }
    
    public void setSACCADE_PROGRAM_INTERVAL(int SACCADE_PROGRAM_INTERVAL)
    {
        this.SACCADE_PROGRAM_INTERVAL = SACCADE_PROGRAM_INTERVAL;
    }
    
    public int getMAX_POSSIBLE_ID_TIME()
    {
        return MAX_POSSIBLE_ID_TIME;
    }
    
    public void setMAX_POSSIBLE_ID_TIME(int MAX_POSSIBLE_ID_TIME)
    {
        this.MAX_POSSIBLE_ID_TIME = MAX_POSSIBLE_ID_TIME;
    }
    
    public int getMAX_SACCADE_PROG_TIME()
    {
        return MAX_SACCADE_PROG_TIME;
    }
    
    public void setMAX_SACCADE_PROG_TIME(int MAX_SACCADE_PROG_TIME)
    {
        this.MAX_SACCADE_PROG_TIME = MAX_SACCADE_PROG_TIME;
    }
    
    public int getMAX_POSSIBLE_ATTEND_TIME()
    {
        return MAX_POSSIBLE_ATTEND_TIME;
    }
    
    public void setMAX_POSSIBLE_ATTEND_TIME(int MAX_POSSIBLE_ATTEND_TIME)
    {
        this.MAX_POSSIBLE_ATTEND_TIME = MAX_POSSIBLE_ATTEND_TIME;
    }
    
    public int getSACCADE_TIME_PENALTY()
    {
        return SACCADE_TIME_PENALTY;
    }
    
    public void setSACCADE_TIME_PENALTY(int SACCADE_TIME_PENALTY)
    {
        this.SACCADE_TIME_PENALTY = SACCADE_TIME_PENALTY;
    }
    
    public int getMIN_ATTEND_TIME_WITH_PENALTIES()
    {
        return MIN_ATTEND_TIME_WITH_PENALTIES;
    }
    
    
    /** Getter for property MAX_ATTEND_TIME_WITH_PENALTIES.
     * @return Value of property MAX_ATTEND_TIME_WITH_PENALTIES.
     *
     */
    public int getMAX_ATTEND_TIME_WITH_PENALTIES()
    {
        return MAX_ATTEND_TIME_WITH_PENALTIES;
    }
    
    public int getNumberOfStates()
    {
        return this.numStates;
    }
    
    public ReadingStateFactory getStateFactory()
    {
        return this.readingStateFactory;
    }
    
    public State getStartingState()
    {
        if (setStartingState != null) 
        {
            return setStartingState; 
        }
        ReadingState startingState = readingStateFactory.createNewReadingState();
        startingState.setAttendedWordID(0);
        startingState = attendCurrentWord(startingState);
        startingState.setTimeAttending(0);
        
        return startingState;
    }
    
    public State getRandomState()
    {
        throw new RuntimeException("getRandomState() is not implemented for sentence world due to large numbers of states... to be implemented soon.");
//        int i = (int) (randomNumberGenerator.nextDouble() * (double) (numStates));
//        return (State) getStateList().get(i);
    }
    public List getStateList()
    {
        throw new RuntimeException("getStateList() is not implemented for sentence world due to large numbers of states... to be implemented soon.");
//        int i = (int) (randomNumberGenerator.nextDouble() * (double) (numStates));
//        return (State) getStateList().get(i);
    }
    
}
