package org.eyelanguage.rl.reading;

import net.pakl.rl.*;
import java.util.*;


/**
 *
 * Inherits from ReadingState, but several methods have been altered so that the model
 * of the environment is word-centric instead of absolute in the sentence (this allows for
 * generalization to new circumstances).
 *
 */
public class ReadingStateRelative extends ReadingState implements HasVectorRepresentation, SubdivisionIdentification
{
    static final long serialVersionUID = -7046034796366757136L;
    
    public ReadingStateRelative(ReadingStateRelative rs)
    {
        super( (ReadingState) rs);
    }
    
    public ReadingStateRelative()
    {
        super();
    }
    
    
    /** Any two objects which are equal must have equal HashCodes, but not necessarily the other way around. */
    public int hashCode()
    {
        if (isIdentified() == true)
            return  1 +
                    + 10 * getDistanceFromAttendCenter()
                    + 1000 * getTimeAttending()
                    + 100000 * getSaccadeRequestDistance()
                    + 10000000 * getTimeInProgrammingSaccade();
        
        return  2 +
                + 10 * getDistanceFromAttendCenter()
                + 1000 * getTimeAttending()
                + 100000 * getSaccadeRequestDistance()
                + 10000000 * getTimeInProgrammingSaccade();
    }
    
    public boolean equals(Object otherObject)
    {
        if (otherObject instanceof ReadingStateRelative)
        {
            ReadingStateRelative otherState = (ReadingStateRelative) otherObject;
            if (otherState.isIdentified() != this.isIdentified())               return false;
            if (otherState.isProgrammingSaccade() != this.isProgrammingSaccade()) return false;
            if (otherState.getTimeAttending() != this.getTimeAttending())       return false;
            if (otherState.getTimeInProgrammingSaccade() != this.getTimeInProgrammingSaccade()) return false;
            if (otherState.getSaccadeRequestDistance() != this.getSaccadeRequestDistance())     return false;
            if (otherState.getAttendedWordLen() != this.getAttendedWordLen())   return false;
            //if (otherState.getAttendedWordPredictability() != this.getAttendedWordPredictability()) return false;
            if (otherState.getDistanceFromAttendCenter() != this.getDistanceFromAttendCenter()) return false;
            if (otherState.getPreviousWordLen() != this.getPreviousWordLen()) return false;
            if (otherState.getNextWordLen() != this.getNextWordLen()) return false;
            return true;
        }
        return false;
    }
    
    public double [] doubleRepresentation()
    {
        double [] result = new double[10];
        result[0] = (double) (500+getTimeAttending()) / 1000.0d;
        result[1] = (double) (50+getDistanceFromAttendCenter()) / 100.0d;
        result[2] = (double) getTimeInProgrammingSaccade() / 20.0d;
        result[3] = (double) (10+getSaccadeRequestDistance()) / 20.0d;
        if (isProgrammingSaccade())
            result[4] = 0.7d;
        else
            result[4] = 0.3d;
        
        result[5] = (double) getAttendedWordLen() / 9.0d;
        result[6] = (double) getPreviousWordLen() / 9.0d;
        result[7] = (double) getNextWordLen() / 9.0d;
        if (isIdentified())
            result[8] = 0.7d;
        else
            result[8] = 0.3d;
        result[9] = ((double)Math.abs(getDistanceFromAttendCenter())) / 100.0d;
        for (int i = 0; i < result.length; i++)
        {
            if (result[i] < 0) { throw new RuntimeException("doublerepresentation index " + i + " was negative."); }
            if (result[i] > 1) { throw new RuntimeException("doublerepresentation index " + i + " was > 1."); }
        }
        return result;
    }
    
    public List binaryRepresentationIndices()
    {
        int [] b = binaryRepresentation();
        ArrayList <Integer> indices = new ArrayList<Integer>();
        for (int i = 0; i < b.length; i++)
        {
            if (b[i] == 1) indices.add(i);
        }
        return indices;
    }
    
    public int [] binaryRepresentation()
    {
        int result [] = null;
        try
        {
            int [] a = representAsVector(getTimeAttending(), -250, +250, 5);
            int [] b = representAsVector(getDistanceFromAttendCenter(), -40, +40, 1);
            int [] c = representAsVector(getTimeInProgrammingSaccade(), 0,  25, 5);
            int [] d = representAsVector(getSaccadeRequestDistance(), -10, 10, 1);
            int [] e = representAsVector(getAttendedWordLen(), 1, 10, 1);
            int [] f = representAsVector(getPreviousWordLen(), 0, 10, 1);
            int [] g = representAsVector(getNextWordLen(), 0, 10, 1);
            int [] h = new int[2];
            if (isIdentified()) 
            {
                h[0] = 1;
            }
            else
            {
                h[1] = 1;
            }

            // getAttendedWordPredictability();
            // isProgrammingSaccade();
            result = concatenateArrays(a,b,c,d,e,f,g,h);
        }
        catch (ArrayIndexOutOfBoundsException e)
        {
            e.printStackTrace();
            throw new RuntimeException(
                    "\n\n"+
                    " *** Bounds for vector representation are too\n"+
                    "     small for binary representation. See line\n" +
                    "     number of binaryRepresentation() call ABOVE ***\n\n");
        }
        return result;
    }
    
    
    public String toString()
    {
        return "STATEREL: [attDst="+this.getDistanceFromAttendCenter()
        + ",attTIM="+this.getTimeAttending()
        + ",sacPrg="+this.isProgrammingSaccade()
        + ",sacPrgTIM="+this.getTimeInProgrammingSaccade()
        + ",sacDst="+this.getSaccadeRequestDistance()
        + ",ID'd?="+this.isIdentified()
//        +",pred="+this.getAttendedWordPredictability()
        +",attlen="+this.getAttendedWordLen()
        +",prevlen="+this.getPreviousWordLen()
        +",nextlen="+this.getNextWordLen()
        +",fixWid="+this.getFixatedWordID()
        +",attWid="+this.getAttendedWordID()
        +"] (eyepos="+this.getEyePosition()+")";
    }
    
    public void parseString(String s)
    {
        String t = s.replaceAll("\\[", "");
        t = t.replaceAll("\\]", "");
        t = t.replaceAll(" ", ",");
        t = t.replaceAll("\\(", ",");
        t = t.replaceAll("\\)", ",");
        StringTokenizer st = new StringTokenizer(t, ",", false);
        while (st.hasMoreTokens())
        {
            StringTokenizer piece = new StringTokenizer(st.nextToken(), "=", false);
            String field = null;
            String value = null;
            if (piece.hasMoreTokens()) field = piece.nextToken();
            if (piece.hasMoreTokens()) value = piece.nextToken();
            if (field != null)
            {
                if (field.equals("attDst"))
                { this.setDistanceFromAttendCenter(intValue(value)); }
                if (field.equals("attTIM"))
                { this.setTimeAttending(intValue(value)); }
                if (field.equals("sacPrg"))
                { this.setProgrammingSaccade(boolValue(value)); }
                if (field.equals("sacPrgTIM"))
                { this.setTimeInProgrammingSaccade(intValue(value)); }
                if (field.equals("sacDst"))
                { this.setSaccadeRequestDistance(intValue(value)); }
                if (field.equals("ID'd?"))
                { this.setIdentified(boolValue(value)); }
                if (field.equals("pred"))
                { this.setAttendedWordPredictability(doubleValue(value)); }
                if (field.equals("attlen"))
                { this.setAttendedWordLen(intValue(value)); }
                if (field.equals("prevlen"))
                { this.setPreviousWordLen(intValue(value)); }
                if (field.equals("nextlen"))
                { this.setNextWordLen(intValue(value)); }
                if (field.equals("fixWid"))
                { this.setFixatedWordID(intValue(value)); }
                if (field.equals("attWid"))
                { this.setAttendedWordID(intValue(value)); }
                if (field.equals("eyepos"))
                { this.setEyePosition(intValue(value)); }
            }
        }
    }
    
    protected int intValue(String s)
    {
        return new Integer(s).intValue();
    }
    
    protected double doubleValue(String s)
    {
        return new Double(s).doubleValue();
    }
    
    protected boolean boolValue(String s)
    {
        if (s.equalsIgnoreCase("true")) return true;
        return false;
    }
    
    public boolean test()
    {
        ReadingStateRelative state1 = new ReadingStateRelative();
        String string1 = "STATEREL: [attDst=0,attTIM=-5,sacPrg=true,sacPrgTIM=10,sacDst=1,ID'd?=false,pred=0.1,attlen=1,prevlen=0,nextlen=3,fixWid=0,attWid=0] (eyepos=0)";
        state1.parseString(string1);
        
        String testString = state1.toString();
        if (!testString.equals(string1))
        {
            System.out.println("[FAIL] Test failed: \noriginal:" + string1 +"\nreturned:" + testString);
        }
        
        System.out.println("[OK] ReadingStateRelative test complete.");
        return true;
    }
    
    public static void main(String args[])
    {
        new ReadingStateRelative().test();
    }
    
    public String getPatchName()
    {
        return this.previousWordLen + "-" + this.attendedWordLen + "-" + this.nextWordLen;
    }
    
}
