package org.eyelanguage.rl.reading;

import net.pakl.rl.*;

import java.util.List;
import java.util.ArrayList;
import java.util.StringTokenizer;

/**
 * In parallel states, we have to keep track of which words of the sentence have been identified
 * already but for default compatability with earlier states we return information about the first word in the
 * window for length, ID and whether the word is identified. Also we need to know how much time has been spent attending to (identifying) each word. */

public class ReadingStateParallelRelative extends ReadingStateRelative implements SubdivisionIdentification
{
    static final long serialVersionUID = 8711529768299836928L;
    public static final boolean FORCE_ROUNDING_TO_GRANULARITY = false;

    
    public boolean [] attWindowIdentified;
    public int [] attWindowWordIDs;
    public int [] attWindowTimes;
    public double [] attWindowPredictabilities;
    public int [] attWindowLengths;
    
    public boolean leftMostAttendedWordIsIdentified()
    {
            return attWindowIdentified[0];
    }
    
    public int getCenterOfAttentionID()
    {
        return attWindowWordIDs[SentenceWorldParallel.CENTER_OF_ATTENTION];
    }
    
    public boolean isIdentifiedInWindow(int index)
    {
        if ((index < 0) || (index > attWindowIdentified.length))
        {
            return true;
        }
        else
        {
            return attWindowIdentified[index];
        }
    }
    
    public int getTimeAttending()
    {
        return attWindowTimes[0];
    }
    
    public void setTimeAttending(int timeAttending)
    {
        
        // attWindowTimes[SentenceWorldParallel.CENTER_OF_ATTENTION] = timeAttending;
    }
    
    public void setParallelTimeAttended(int i, int value)
    {
        attWindowTimes[i] = value;
    }


    public void incParallelTimeAttended(int i, int delta)
    {
        setParallelTimeAttended(i, getParallelTimeAttended(i)+delta);
    }
        
    public int getParallelTimeAttended(int i)
    {
        return attWindowTimes[i];
    }
    

    public void setAttendedWordID()
    {
        //throw new RuntimeException("Not used in parallel states.");
    }

    public int getAttendedWordID()
    {
        return attWindowWordIDs[0];
    }

    public int getBeginningOfAttWindow()
    {
        return attWindowWordIDs[0];
    }
    
    public ReadingStateParallelRelative()
    {
        attWindowIdentified = new boolean[SentenceWorldParallel.PARALLEL_ATTENDABLE_WORDS];
        attWindowWordIDs = new int[SentenceWorldParallel.PARALLEL_ATTENDABLE_WORDS];
        attWindowTimes = new int[SentenceWorldParallel.PARALLEL_ATTENDABLE_WORDS];
        attWindowPredictabilities = new double[SentenceWorldParallel.PARALLEL_ATTENDABLE_WORDS];
        attWindowLengths = new int[SentenceWorldParallel.PARALLEL_ATTENDABLE_WORDS];
    }
    
    public ReadingStateParallelRelative(ReadingStateParallelRelative s)
    {
        attWindowIdentified = new boolean[SentenceWorldParallel.PARALLEL_ATTENDABLE_WORDS];
        attWindowWordIDs = new int[SentenceWorldParallel.PARALLEL_ATTENDABLE_WORDS];
        attWindowTimes = new int[SentenceWorldParallel.PARALLEL_ATTENDABLE_WORDS];
        attWindowPredictabilities = new double[SentenceWorldParallel.PARALLEL_ATTENDABLE_WORDS];
        attWindowLengths = new int[SentenceWorldParallel.PARALLEL_ATTENDABLE_WORDS];
        copyFrom(s);
    }

    public int getSumAttWindowTimes()
    {
        int sum = 0;
        for (int i = 0; i < attWindowTimes.length; i++)
        {
            sum = sum + attWindowTimes[i];
        }
        return sum;
    }
    
    public int getNumIdentified()
    {
        int sum = 0;
        for (int i = 0; i < attWindowIdentified.length; i++)
        {
            if (attWindowIdentified[i] == true) { sum++; }
        }
        return sum;
    }
    
    public boolean isIdentified()
    {
        return attWindowIdentified[0];
    }

    public void copyFrom(ReadingState s)
    {
        ReadingStateParallelRelative toCopyFrom = (ReadingStateParallelRelative) s;

        this.setDistanceFromAttendCenter(toCopyFrom.getDistanceFromAttendCenter());
        this.setEyePosition(toCopyFrom.getEyePosition());
        this.setFixatedWordID(toCopyFrom.getFixatedWordID());
        this.setProgrammingSaccade(toCopyFrom.isProgrammingSaccade());
        this.setSaccadeRequestDistance(toCopyFrom.getSaccadeRequestDistance());
        this.setTimeInProgrammingSaccade(toCopyFrom.getTimeInProgrammingSaccade());
        this.setNextWordLen(toCopyFrom.getNextWordLen());
        this.setPreviousWordLen(toCopyFrom.getPreviousWordLen());
        
        for (int i = 0; i < attWindowIdentified.length; i++)
        {
            this.attWindowIdentified[i] = toCopyFrom.attWindowIdentified[i];
            this.attWindowWordIDs[i] = toCopyFrom.attWindowWordIDs[i];
            this.attWindowTimes[i] = toCopyFrom.attWindowTimes[i];
            this.attWindowPredictabilities[i] = toCopyFrom.attWindowPredictabilities[i];
        }

        for (int i = 0; i < attWindowLengths.length; i++)
        {
            this.attWindowLengths[i] = toCopyFrom.attWindowLengths[i];
        }
    }


    
    public int getAttendedWordLen()
    {
        return attWindowLengths[0];
    }

    
    
    public boolean equals(Object otherObject)
    {
        if (otherObject instanceof ReadingStateParallelRelative)
        {
            ReadingStateParallelRelative otherState = (ReadingStateParallelRelative) otherObject;
            if (otherState.isProgrammingSaccade() != this.isProgrammingSaccade()) return false;
            if (otherState.getTimeInProgrammingSaccade() != this.getTimeInProgrammingSaccade()) return false;
            if (otherState.getSaccadeRequestDistance() != this.getSaccadeRequestDistance())     return false;
            if (otherState.getDistanceFromAttendCenter() != this.getDistanceFromAttendCenter()) return false;

            for (int i = 0; i < attWindowLengths.length; i++)
            {
                if (this.attWindowLengths[i] != otherState.attWindowLengths[i]) return false;
            }
            for (int i = 0; i < attWindowIdentified.length; i++)
            {
                if (this.attWindowIdentified[i] != otherState.attWindowIdentified[i]) return false;
                if (this.attWindowTimes[i] != otherState.attWindowTimes[i]) return false;
            }
            if (this.getNextWordLen() != otherState.getNextWordLen()) { return false; }
            if (this.getPreviousWordLen() != otherState.getPreviousWordLen()) { return false; }
            return true;
            
        }
        return false;
    }
    
    public static int roundTo(int value, int roundToNearest)
    {
        return (int) roundTo((double) value, (double)roundToNearest);
    }
    
    
    public static double roundTo(double value, double roundToNearest)
    {
        return Math.round(value * (1.0 / roundToNearest)) / (1.0 / roundToNearest);
    }
    
    public void roundToLinearAttentionGradient()
    {
        for (int i = 0; i < attWindowIdentified.length; i++)
        {
            int wordIDToProcess = attWindowWordIDs[i];
            
            int granularity = 0;
            
            if (wordIDToProcess == this.getCenterOfAttentionID()) granularity = 5;
            if (Math.abs(wordIDToProcess-this.getCenterOfAttentionID()) == 1) granularity = 3;
            if (Math.abs(wordIDToProcess-this.getCenterOfAttentionID()) == 2) granularity = 2;
            if (Math.abs(wordIDToProcess-this.getCenterOfAttentionID()) == 3) granularity = 1;     
            
            attWindowTimes[i] = roundTo(attWindowTimes[i], granularity);
        }
    }
    
    /** Any two objects which are equal must have equal HashCodes, but not necessarily the other way around. */
    public int hashCode()
    {
        
        if (identified == true)
            return  1 +
                    + 10 * getDistanceFromAttendCenter()
                    + 1000 * getTimeAttending() 
                    + 10000 * this.getSumAttWindowTimes()
                    + 100000 * getSaccadeRequestDistance()
                    + 10000000 * getTimeInProgrammingSaccade();
        
        return  2 +
                + 10 * getDistanceFromAttendCenter()
                + 1000 * getTimeAttending()
                + 10000 * this.getSumAttWindowTimes()
                + 100000 * getSaccadeRequestDistance()
                + 10000000 * getTimeInProgrammingSaccade();
    }
    
    public ReadingStateParallelRelative fromString(String s)
    {
        ArrayList<String> required = new ArrayList<String>();
        required.add("ecc");
        required.add("fixID");
        required.add("eyepos");
        required.add("sacPrg");
        required.add("sacReqDst");
        required.add("sacPrgTime");
        required.add("attIDs");
        required.add("attLens");
        required.add("attTimes");
        required.add("identified");
        StringTokenizer t = new StringTokenizer(s, ",");
        while (t.hasMoreTokens())
        {
            StringTokenizer equality = new StringTokenizer(t.nextToken(),"=");
            String key = equality.nextToken();
            String value = equality.nextToken();
            
            required.remove(key);
            
            if (key.equals("ecc"))          
                setDistanceFromAttendCenter(new Integer(value).intValue());
            if (key.equals("fixID"))       
                setFixatedWordID(new Integer(value).intValue());
            if (key.equals("eyepos"))       
                setEyePosition(new Integer(value).intValue());
            if (key.equals("sacPrg"))     
            {
                if (value.equals("true")) 
                    { setProgrammingSaccade(true); } 
                else 
                    { setProgrammingSaccade(false); }
            }
            if (key.equals("sacReqDst"))    
                setSaccadeRequestDistance(new Integer(value).intValue());
            if (key.equals("sacPrgTime"))   
                setTimeInProgrammingSaccade(new Integer(value).intValue());
            if (key.equals("attIDs"))       
                plop(attWindowWordIDs, value);
            if (key.equals("attLens"))      
                plop(attWindowLengths, value);
            if (key.equals("attTimes"))     
                plop(attWindowTimes, value);
            if (key.equals("identified"))
            {
                for (int i = 0; i < value.length(); i++)
                {
                    if (value.charAt(i) == 'x') 
                        { attWindowIdentified[i] = true; } 
                    else 
                        { attWindowIdentified[i] = false; }
                }
            }
        }
        
        if (!required.isEmpty())
        {
            throw new RuntimeException("\nNot all keys were specified for instantiating a class from a string.\n"
                    +"Remaining keys: " + required);
        }
        System.out.println("State successfully instantiated from String: " + toString());
        return this;
    }

    private void plop(int [] array, String values)
    {
        StringTokenizer data = new StringTokenizer(values,"|");
        int slot = 0;
        while (data.hasMoreTokens())
        {
            array[slot] = new Integer(data.nextToken()).intValue();
            slot++;
        }
    }
    
    public String toString()
    {
        String result = "STATEPARREL: [ecc="+this.getDistanceFromAttendCenter()
        + ",sacPrg="+this.isProgrammingSaccade()
        + ",sacPrgTime="+this.getTimeInProgrammingSaccade()
        + ",sacReqDst="+this.getSaccadeRequestDistance()
        +",attLens=";
        
        for (int i = 0; i < this.attWindowLengths.length; i++)
        {
            result+=attWindowLengths[i]+"|";
        }
        result+=",attIDs=";
                
        for (int i = 0; i < this.attWindowWordIDs.length; i++)
        {
            result+=attWindowWordIDs[i]+"|";
        }
        result += ",eyepos="+this.getEyePosition()
        +",fixID="+this.getFixatedWordID()
        +",startAttID="+this.getBeginningOfAttWindow()
        +",centerAttID="+(this.getBeginningOfAttWindow()+SentenceWorldParallel.CENTER_OF_ATTENTION)
        +",endAttWinID="+(this.getBeginningOfAttWindow()+SentenceWorldParallel.PARALLEL_ATTENDABLE_WORDS-1)
        +",attTimes=";

        for (int i = 0; i < attWindowTimes.length; i++)
        {
            result += attWindowTimes[i] + "|";
        }
        result +=",identified=";
        for (int i = 0; i < attWindowIdentified.length; i++)
        {
            if (attWindowIdentified[i])
            {
                result += "x"; 
            }
            else
            {
                result += "_";
            }
        }
        result +="]";
        return result;
    }
    
    public String getPatchName()
    {
        String result="";
        for (int i = 0; i < attWindowLengths.length; i++)
        {
            result += attWindowLengths[i];
            if (i < attWindowLengths.length-1) { result += "-"; }
        }
        return result;
    }    
    
    


}
