package org.eyelanguage.rl.reading;
import net.pakl.rl.*;
import java.util.*;
import java.io.*;
import java.util.zip.*;

/** A class configured by a textfile called "reading.prop" that can be run from
 * the command line to initiate a
 * reinforcement learning simulation on a reading problem;  all variables below
 * can be specified in the reading.prop file, e.g. saccadicError = true */
public class ReadingMain
{
    private static final String VERSION = "20060203-1631";
    
    /** reading.prop parameter: How many interations to train before
     * stopping and testing. */
    protected int totalIterations = 2000;
    protected int testAgentEvery = 0;
    
    // ----------------------------------------------------------------------------------------------------------------
    // These specify the "training" corpus.
    // ----------------------------------------------------------------------------------------------------------------
    /** reading.prop parameter: A visual representation of the sentence content
     * to be constructed, e.g. 'x xxx xxxxx'. */
    
    /** Function Approximators: update incrementally (on each input-output mapping) if true, else epoch-wise. */
    protected boolean incrementalUpdate = true;
    /** reading.prop parameter: Serial attention means attend one word at a time, semiparallel means multiple but with
     * a serial sliding window, and parallel means fully parallel processing of words in any order. */
    protected String attention = "serial";        // serial, semiparallel, or parallel eventually.
    protected boolean PARALLEL_UNIFORM_ATTENTION = false;
    protected boolean PARALLEL_LINEAR_ATTENTION_GRADIENT = false;
    protected boolean PARALLEL_DIVIDED_ATTENTION_GRADIENT = false;
    
    /** reading.prop parameter: True if saccadic motor error should be enabled, false otherwise. */
    protected boolean saccadicError = false;
    /** reading.prop parameter: The amount of "spread" of this particular kind of saccadic error (e.g. could be the standard deviation of a normal distribution,
     * or the width of a uniform distribution.) */
    protected double saccadicErrorSpread = 0;
    /** reading.prop parameter: If true, one extra timestemp of processing is required for each character space from the center of the word being processed
     * for word identification.  If false, a constant penalty (default 1) is added no matter how eccentric the eyes are.*/
    protected boolean acuityLinearPenalize = false;

    /** Amount of cost added per character eccentricity for linear acuity penalty. */
    protected int acuityLinearPenaltySlope = 5;
    /** reading.prop parameter: How much time (negative) should be removed during a saccadic eye movement? */
    protected int saccadeTimePenalty = -5;
    /** reading.prop parameter: Should the Gaussian saccadic error be used -- otherwise, a uniform saccadic error will be used. */
    protected boolean saccadicErrorGaussian = true;
    /** The reward received by the agent when entering a state where the current word is newly identified. */
    protected double identifyReinforcement = 1.0;
    /** The reward received by the agent when entering a terminal state. */
    protected double reachTerminalReinforcement = 0.0;    // Should be zero so that reward doesn't get infinite at the end of the trial.
    /** reading.prop parameter */
    protected double defaultReinforcement = -1.0;
    /** Amount of time it takes the agent to program a saccade. */
    protected int saccPrgTime = 15;
    /** If attention is not serial, The number of words the agent can attend (process lexically) in parallel to the right of attention focus. */
    protected int parallelAttendableWords = 2;
    /** If attention is not serial, The number of words the agent can attend (process lexically) in parallel to the left of attention focus. */
    protected int centerOfAttention = 0;
    /** The maximum allowed distance in the state space between fovea and center of attention. */
    protected int acuityLimit = 1000;
    /** Specifies in generic policy if saccade request actions and attention
     * shift actions can occur at the same time. */
    protected boolean ALLOW_SIMULTANEOUS_SACREQ_AND_ATTNEXT = false;
    /** Specifies in generic policy if attention shifts and word identification can occur at the same time. */
    protected boolean ALLOW_SIMULTANEOUS_ATTNEXT_AND_IDENT = false;
    
    /** Affect PolicyExtractor to load value functions if the state does not reside in the current value function; if the state does not support SubdivisionIdentification the PolicyExtractor will communicate this. */
    protected boolean DYNAMICALLY_LOAD_VALUE_FUNCTION_PATCHES = false;
    /** Stores a name in the value function that should correspond to the patch name identifying the patch name; perhaps in the future value function patches should have more than one name.  */
    protected String valueFunctionName = "valueFunction";

    
    
    
    protected String sentence = "a funny cat";
    /** reading.prop parameter: Space-separated real numbers specifying the
     * predictability of each word, starting with the second word in the
     * sentence? */
    protected String predictabilities = "0.5 0.2";
    /** Specified the amount of time it will take the agent to recognize each
     * word, if that word if foveated in its center. */
    protected String aidTimes = "30 30 30";    
    protected String wordLengths = "";
    protected boolean wordLengthsOverridden = false;
    
    // ----------------------------------------------------------------------------------------------------------------
    // These specify the "testing" corpus.
    // ----------------------------------------------------------------------------------------------------------------
    /** reading.prop parameter */
    protected String testSentence = "a funny cat";
    /** reading.prop parameter */
    protected int testIterations = 1;
    /** reading.prop parameter */
    protected String testPredictabilities = "0.5 0.2";
    /** reading.prop parameter */
    protected String testAidTimes = "30 30 30";
    /** reading.prop parameter */
    protected int hiddenLayerSize = 10;
    protected String testWordLengths = "";

    protected boolean testWordLengthsOverridden = false;
    
    
    // ----------------------------------------------------------------------------------------------------------------
    // Logistics of Implementation / Execution
    // ----------------------------------------------------------------------------------------------------------------
    /** reading.prop parameter */
    protected String agentType = "serial";                // If parallel, set agentParallelNumThreads
    /** reading.prop parameter */
    protected int agentParallelNumThreads = 2;            // Only applies of agentType = parallel
    /** reading.prop parameter */
    protected int boundedBufferSize = 0;
    /** reading.prop parameter */
    protected double valueOfTerminalStates = 0.0;
    /** reading.prop parameter */
    protected boolean testTrainSameVf = false;             // Should testing and training be done on the same valueFunction?
    /** reading.prop parameter */
    protected boolean stateActionCache = false;
    /** reading.prop parameter */
    protected boolean bitPacker = false;
    /** reading.prop parameter */
    protected String valueFunctionToLoad = "";

    /** reading.prop parameter */
    protected boolean wrapSerialValueFunctionWithParallelAdapter = false;
    /** reading.prop parameter */
    protected String valueFunctionToSave = "";
    /** reading.prop parameter */
    protected String loadWorldCache = "";
    /** reading.prop parameter */
    protected boolean distributedJavaParty = false;
    
    /** (Experimenal) Engages special handcrafted code for function approximator state vector representation. */
    protected boolean handCraftedVectors = false;
    
    // ----------------------------------------------------------------------------------------------------------------
    // Reinforcement Learning parameters
    // ----------------------------------------------------------------------------------------------------------------
    /** reading.prop parameter */
    protected String rlAlgorithm = "valueIteration";
    /** This number is used in the TrajectorySampler mode, and specified how
     * likely the agent is to take the best known action.  Lower values
     * encourage more of the random exploration behavior. */
    protected double greed = 1.0d;
    /** Represents how much future reward is weighted in comparison to the
     * immediate reward.  If your environment has loops (and the agent can
     * return to the same state it was before) you may wish to set this to a
     * value less than 1 to avoid exploding value functions (i.e. infinite
     * predicted reward). */
    protected double discountFactor = 1.0;
    /** Amount by which to incrementally update current predictions, useful in the face of stochasticity. */
    protected double epsilon = 1.0;
    /** Determines the learning rate of the neural network or function approximator that is being used. */
    protected double learningRate = 0.01;
    /** Determines how recent changes in the neural network or function approximator continue to affect the network. */
    protected double momentum = 0.0;
    /** For a neural network or function approximator, specifies the meaning of
     * the maximal output and minimal output values. */
    protected double netOutputRange = 150.0d;
    /** reading.prop parameter */
    protected int saveNetworkEvery = 0;
    /** reading.prop parameter */
    protected String readingStateType = "Absolute";
    /** reading.prop parameter */
    protected long randomSeed = 123456L;
    
    protected double valueOfNonStoredStates = 0;
    
    protected String startState = "";
    
    protected boolean updateOnlyWhenGreedy = true;
    
    /** reading.prop parameter */
    protected boolean residualAlgorithm = true;
    /** reading.prop parameter */
    protected double residualWeighting = 0.5;
    /** reading.prop parameter */
    String valueFunctionType = "LookupTable";  // LookupTable  or ResidualAlgorithmLinear
    /** reading.prop parameter */
    boolean valueFunctionSafety = true;
    /** reading.prop parameter */
    boolean valueFunctionStateBounds = false;
    /** reading.prop parameter */
    protected boolean stateVectorsCrossProduct = false;
    /** reading.prop parameter */
    protected boolean stateVectorsComplement = false;
    
    public static void main(String args[])
    {
        new ReadingMain();
    }
    
    public ReadingMain()
    {
        loadProperties();
        showProperties();
        trainSingleSentence();
    }
    
    
    private SentenceWorld configureSentenceWorld()
    {
        SentenceWorld sWorld = null;
        if (attention.equalsIgnoreCase("serial")) sWorld = new SentenceWorld("trainCorpus", randomSeed);
        if (attention.equalsIgnoreCase("semiparallel")) sWorld = new SentenceWorldParallel("trainCorpus", randomSeed);
        if (sWorld == null)
        {
            throw new RuntimeException("Sorry, attention must be either serial or semiparallel... but was " + attention);
        }
        if (readingStateType.equalsIgnoreCase("Relative"))
        {
            System.out.println("! SETTING UP RELATIVE READING STATES");
            if (attention.equalsIgnoreCase("serial"))
            {
                sWorld.setupReadingStateFactory(ReadingStateFactory.RELATIVE_READING_STATES);
            }
            else
            {
                sWorld.setupReadingStateFactory(ReadingStateFactory.RELATIVE_PARALLEL_READING_STATES);
                
                SentenceWorldParallel asParallel = (SentenceWorldParallel) sWorld;
                asParallel.setPARALLEL_ATTENDABLE_WORDS(parallelAttendableWords);
                asParallel.setCENTER_OF_ATTENTION(centerOfAttention);
                asParallel.UNIFORM_ATTENTION = this.PARALLEL_UNIFORM_ATTENTION;
                asParallel.LINEAR_ATTENTION_GRADIENT = this.PARALLEL_LINEAR_ATTENTION_GRADIENT;
                asParallel.DIVIDED_ATTENTION = this.PARALLEL_DIVIDED_ATTENTION_GRADIENT;
            }
        }
        sWorld.setMAX_SACCADE_PROG_TIME(saccPrgTime);
        sWorld.SACCADIC_ERROR = this.saccadicError;
        sWorld.SACCADIC_ERROR_GAUSSIAN = this.saccadicErrorGaussian;
        sWorld.SACCADIC_ERROR_SPREAD = this.saccadicErrorSpread;
        sWorld.ACUITY_LINEAR_PENALIZE = this.acuityLinearPenalize;
        sWorld.ACUITY_LINEAR_PENALTY_SLOPE = this.acuityLinearPenaltySlope;
        sWorld.SACCADE_TIME_PENALTY = this.saccadeTimePenalty;
        sWorld.ACUITY_LIMIT = this.acuityLimit;
        if (!startState.equals(""))
        {
            sWorld.setStartingState = (new ReadingStateParallelRelative().fromString(startState));
        }
        return sWorld;
    }
    
    
    private void trainSingleSentence()
    {
        long start = System.currentTimeMillis();
        SentenceWorld sWorld = configureSentenceWorld();
        sWorld.setSentence(sentence, predictabilities, aidTimes, wordLengths, wordLengthsOverridden);

        if (rlAlgorithm.equalsIgnoreCase("valueIteration"))
        {
            if (!sWorld.getIsBuilt()) sWorld.build();         // Create all states
        }    
        ReadingPolicy readingPolicy = Toolbox.createSimpleReadingPolicy(sWorld);
        readingPolicy.setALLOW_SIMULTANEOUS_SACREQ_AND_ATTNEXT(ALLOW_SIMULTANEOUS_SACREQ_AND_ATTNEXT);
        readingPolicy.setALLOW_SIMULTANEOUS_ATTNEXT_AND_IDENT(ALLOW_SIMULTANEOUS_ATTNEXT_AND_IDENT);
        
        ReadingReinforcementFunction rf = null;
        if (attention.equalsIgnoreCase("serial"))
        {
            rf = new ReadingReinforcementFunction(sWorld);
        }
        else
        {
            rf = new ReadingParallelReinforcementFunction(sWorld);
        }
        rf.setDefaultReinforcement(defaultReinforcement);
        rf.setRewardForSuccessfulIdentification(identifyReinforcement);
        rf.setReinforcementsForActionsOntoTerminalState(reachTerminalReinforcement);               // Just like in FrogWorld's Toolbox
        
        Agent reader = null;
        if (agentType.equalsIgnoreCase("parallel"))
        {
            reader = new AgentParallelized("reader", agentParallelNumThreads);
        }
        else
        {
            reader = new Agent("reader");
        }
        reader.printStartStateValueEvery = 100;
        reader.setReinforcementFunction(rf);
        reader.setPolicy(readingPolicy);
        reader.setDiscountFactor(discountFactor);
        reader.setEpsilon(epsilon);

        reader.setGreed(greed);
        reader.UPDATE_ONLY_WHEN_GREEDY = updateOnlyWhenGreedy;
        ValueFunction valueFunction = null;        
        ValueFunction tempNewValueFunction = null;

        if (valueFunctionToLoad.equals(""))
        {
            valueFunction = initializeValueFunction(sWorld);
            if (!testTrainSameVf) tempNewValueFunction = initializeValueFunction(sWorld);
        }
        else
        {
            valueFunction = loadValueFunction(valueFunctionToLoad);
            if ((!testTrainSameVf) && (totalIterations > 0))
            {
                tempNewValueFunction = loadValueFunction(valueFunctionToLoad);
            }
        }

        if (totalIterations > 0)
        {
            valueFunction.setValueOfTerminalStates(valueOfTerminalStates);
            if (valueFunction instanceof ValueFunctionHashMap) 
                ((ValueFunctionHashMap)valueFunction).setValueOfNonStoredStates(valueOfNonStoredStates);
            if (!testTrainSameVf) 
            {
                tempNewValueFunction.setValueOfTerminalStates(valueOfTerminalStates);
                if (tempNewValueFunction instanceof ValueFunctionHashMap)
                ((ValueFunctionHashMap)tempNewValueFunction).setValueOfNonStoredStates(valueOfNonStoredStates);
            }
            
            
            System.out.println("Done with ValueFunction...");
            
            int sampleTimingEvery = 1000;
            reader.setWorld(sWorld);
            int numStates = sWorld.getNumberOfStates();
            double timerStart = (double) System.currentTimeMillis();
            for (int trial = 1; trial < totalIterations; trial++)
            {
                if (receivedEndRunMessage()) break;
                if (testTrainSameVf)
                {
                    trainAgent(reader, valueFunction, valueFunction, trial, numStates);
                }
                else
                {
                    trainAgent(reader, valueFunction, tempNewValueFunction, trial, numStates);
                    ValueFunction temp = tempNewValueFunction;
                    tempNewValueFunction = valueFunction;
                    valueFunction = temp;
                }
                
                if ((testAgentEvery > 0) && (trial % testAgentEvery == 0))
                {
                    testAgent(sWorld, valueFunction);
                }
                
                saveValueFunction(valueFunction, trial);
                
                if (trial % sampleTimingEvery == 0)
                {
                    double timerCease = (double) System.currentTimeMillis();
                    double duration = (timerCease - timerStart) / 60d / 1000d;
                    System.out.println("\n"+sampleTimingEvery+" Iterations took " + duration + " minutes.");
                    System.out.println("Estimated " + ((double)((totalIterations-trial)/sampleTimingEvery) * duration / 60d) + " hours remaining.");
                    if (valueFunction instanceof ValueFunctionHashMap)
                    {
                        System.out.println("size(vf) = "+((ValueFunctionHashMap)valueFunction).getSize());
                    }
                    timerStart = (double) System.currentTimeMillis();
                }
            }
            System.out.println("Iterations completed, stopping run.");
            if (!valueFunctionToSave.equals("")) saveValueFunction(valueFunction, valueFunctionToSave);
            
        }
        testAgent(sWorld, valueFunction);
        long cease = System.currentTimeMillis();
        
        System.out.println("Total run time was " + ((double)cease-(double)start)/(double)1000/(double)3600 + " hours");
    }
    
    private void testAgent(SentenceWorld sWorld, ValueFunction valueFunction)
    {
        System.out.println("Testing Agent");
        SentenceWorld tWorld = sWorld;
        if (!testSentence.equals(""))
        {
            if (attention.equalsIgnoreCase("serial")) tWorld = new SentenceWorld("trainCorpus", randomSeed);
            if (attention.equalsIgnoreCase("semiparallel")) tWorld = new SentenceWorldParallel("trainCorpus", randomSeed);
            
            if (readingStateType.equalsIgnoreCase("Relative"))
            {
                System.out.println("! SETTING UP RELATIVE READING STATES");
                if (attention.equalsIgnoreCase("serial"))
                {
                    tWorld.setupReadingStateFactory(ReadingStateFactory.RELATIVE_READING_STATES);
                }
                else
                {
                    tWorld.setupReadingStateFactory(ReadingStateFactory.RELATIVE_PARALLEL_READING_STATES);
                    
                    SentenceWorldParallel asParallel = (SentenceWorldParallel) tWorld;
                    asParallel.setPARALLEL_ATTENDABLE_WORDS(parallelAttendableWords);
                    asParallel.setCENTER_OF_ATTENTION(centerOfAttention);
                    asParallel.UNIFORM_ATTENTION = this.PARALLEL_UNIFORM_ATTENTION;
                    asParallel.LINEAR_ATTENTION_GRADIENT = this.PARALLEL_LINEAR_ATTENTION_GRADIENT;
                    asParallel.DIVIDED_ATTENTION = this.PARALLEL_DIVIDED_ATTENTION_GRADIENT;
                }
            }
            tWorld.setMAX_SACCADE_PROG_TIME(saccPrgTime);
            tWorld.SACCADIC_ERROR = this.saccadicError;
            tWorld.SACCADIC_ERROR_GAUSSIAN = this.saccadicErrorGaussian;
            tWorld.SACCADIC_ERROR_SPREAD = this.saccadicErrorSpread;
            tWorld.ACUITY_LINEAR_PENALIZE = this.acuityLinearPenalize;
            tWorld.ACUITY_LINEAR_PENALTY_SLOPE = this.acuityLinearPenaltySlope;
            tWorld.SACCADE_TIME_PENALTY = this.saccadeTimePenalty;
            tWorld.ACUITY_LIMIT = this.acuityLimit;
            tWorld.setSentence(testSentence, testPredictabilities, testAidTimes, testWordLengths, testWordLengthsOverridden);
            if (!startState.equals(""))
            {
                tWorld.setStartingState = (new ReadingStateParallelRelative().fromString(startState));
            }
        }
        else
        {
            tWorld = sWorld;
        }
        if (rlAlgorithm.equalsIgnoreCase("valueIteration"))
        {
            if (!tWorld.getIsBuilt()) tWorld.build();         // Create all states
        }
        
        for (int tests = 0; tests < testIterations; tests++)
        {
            System.out.println("Test Number " + tests);
            PolicyExtractor policyExtractor = new PolicyExtractor();
            if (DYNAMICALLY_LOAD_VALUE_FUNCTION_PATCHES) policyExtractor.DYNAMICALLY_LOAD_VALUE_FUNCTION_PATCHES = true;
            ReadingReinforcementFunction rfTest = null;
            if (attention.equalsIgnoreCase("serial"))
            {
                rfTest = new ReadingReinforcementFunction(tWorld);
            }
            else
            {
                rfTest = new ReadingParallelReinforcementFunction(tWorld);
            }
            rfTest.setDefaultReinforcement(defaultReinforcement);
            rfTest.setRewardForSuccessfulIdentification(identifyReinforcement);
            rfTest.setReinforcementsForActionsOntoTerminalState(reachTerminalReinforcement);               // Just like in FrogWorld's Toolbox
            ReadingPolicy readingPolicyTest = Toolbox.createSimpleReadingPolicy(tWorld);
            try
            {
                System.out.println(policyExtractor.extractOptimalPolicy(readingPolicyTest, valueFunction, sWorld, tWorld, rfTest, discountFactor));
            }
            catch (Exception e)
            {
                e.printStackTrace();
            }
        }
    }
    
    
    private ValueFunction initializeValueFunction(SentenceWorld sWorld)
    {
        System.out.println("Initializing a new value function with a "+sWorld.getClass()+".");
        
        if (valueFunctionType.equalsIgnoreCase("ResidualAlgorithmLinear"))
        {
            System.out.println("USING Residual Algorithm -- Linear Approximator!!!");
            ValueFunction valueFunction = new ValueFunctionResidualAlgorithmLinear(sWorld);
            
            ((ValueFunctionResidualAlgorithmLinear) valueFunction).setMaxMinValue(this.netOutputRange);
            ((ValueFunctionResidualAlgorithmLinear) valueFunction).CROSS_PRODUCT = stateVectorsCrossProduct;
            ((ValueFunctionResidualAlgorithmLinear) valueFunction).BINARY_COMPLEMENT = stateVectorsComplement;
            ((ValueFunctionResidualAlgorithmLinear) valueFunction).RESIDUAL_ALGORITHM = residualAlgorithm;
            ((ValueFunctionResidualAlgorithmLinear) valueFunction).INCREMENTAL_UPDATE = incrementalUpdate;
            ((ValueFunctionResidualAlgorithmLinear) valueFunction).HAND_CRAFTED_VECTORS = handCraftedVectors;
            ((ValueFunctionResidualAlgorithmLinear) valueFunction).setLearningRate(this.learningRate);
            ((ValueFunctionResidualAlgorithmLinear) valueFunction).setResidualWeighting(this.residualWeighting);
            valueFunction.setName(valueFunctionName);
            return valueFunction;
        }
        if (valueFunctionType.equalsIgnoreCase("ResidualAlgorithm"))
        {
            System.out.println("USING Residual Algorithm");
            ValueFunction valueFunction = new ValueFunctionResidualAlgorithmPerceptron(sWorld);
            
            ((ValueFunctionResidualAlgorithmPerceptron) valueFunction).setMaxMinValue(this.netOutputRange);
            ((ValueFunctionResidualAlgorithmPerceptron) valueFunction).setLearningRate(this.learningRate);
            ((ValueFunctionResidualAlgorithmPerceptron) valueFunction).setRandomSeed(this.randomSeed);
            
            ((ValueFunctionResidualAlgorithmPerceptron) valueFunction).setResidualWeighting(this.residualWeighting);
            ((ValueFunctionResidualAlgorithmPerceptron) valueFunction).setHiddenUnits(this.hiddenLayerSize);
            ((ValueFunctionResidualAlgorithmPerceptron) valueFunction).INCREMENTAL_UPDATE = this.incrementalUpdate;
            valueFunction.setName(valueFunctionName);
            return valueFunction;
        }
        if (valueFunctionType.equalsIgnoreCase("ResidualAlgorithmBairdPerceptron"))
        {
            System.out.println("USING Residual Algorithm + Baird Perceptron");
            ValueFunction valueFunction = new ValueFunctionResidualAlgorithmBairdPerceptron(sWorld);
            
            ((ValueFunctionResidualAlgorithmPerceptron) valueFunction).setMaxMinValue(this.netOutputRange);
            ((ValueFunctionResidualAlgorithmPerceptron) valueFunction).setLearningRate(this.learningRate);
            ((ValueFunctionResidualAlgorithmPerceptron) valueFunction).setRandomSeed(this.randomSeed);
            
            ((ValueFunctionResidualAlgorithmPerceptron) valueFunction).setResidualWeighting(this.residualWeighting);
            ((ValueFunctionResidualAlgorithmPerceptron) valueFunction).setHiddenUnits(this.hiddenLayerSize);
            ((ValueFunctionResidualAlgorithmPerceptron) valueFunction).INCREMENTAL_UPDATE = this.incrementalUpdate;
            valueFunction.setName(valueFunctionName);
            return valueFunction;
        }
        if (valueFunctionType.equalsIgnoreCase("ResidualAlgorithmPatrykNetwork"))
        {
            System.out.println("USING Residual Algorithm + Patryk Network");
            ValueFunction valueFunction = new ValueFunctionResidualAlgorithmPatrykNetwork(sWorld);
            
            ((ValueFunctionResidualAlgorithmPerceptron) valueFunction).setMaxMinValue(this.netOutputRange);
            ((ValueFunctionResidualAlgorithmPerceptron) valueFunction).setLearningRate(this.learningRate);
            ((ValueFunctionResidualAlgorithmPerceptron) valueFunction).setRandomSeed(this.randomSeed);
            
            ((ValueFunctionResidualAlgorithmPerceptron) valueFunction).setResidualWeighting(this.residualWeighting);
            ((ValueFunctionResidualAlgorithmPerceptron) valueFunction).setHiddenUnits(this.hiddenLayerSize);
            ((ValueFunctionResidualAlgorithmPerceptron) valueFunction).INCREMENTAL_UPDATE = this.incrementalUpdate;
            valueFunction.setName(valueFunctionName);
            return valueFunction;
        }
        
        if (valueFunctionType.equalsIgnoreCase("Perceptron"))
        {
            System.out.println("USING Backprop -- GradientDescent Approximator WITHOUT RESIDUAL ALGORITHMS!!!");
            ValueFunction valueFunction = new ValueFunctionPerceptron(sWorld);
            
            ((ValueFunctionPerceptron) valueFunction).setMaxMinValue(this.netOutputRange);
            ((ValueFunctionPerceptron) valueFunction).setLearningRate(this.learningRate);
            ((ValueFunctionPerceptron) valueFunction).setRandomSeed(this.randomSeed);
            
            ((ValueFunctionPerceptron) valueFunction).setHiddenUnits(this.hiddenLayerSize);
            ((ValueFunctionPerceptron) valueFunction).setMomentum(this.momentum);
            
            ((ValueFunctionPerceptron) valueFunction).INCREMENTAL_UPDATE = this.incrementalUpdate;
            valueFunction.setName(valueFunctionName);
            return valueFunction;
        }
        if (valueFunctionType.equalsIgnoreCase("LookupTable"))
        {
            System.out.println("USING LOOKUP TABLE VALUE FUNCTION.");
            ValueFunctionHashMap vf = new ValueFunctionHashMap(sWorld, valueFunctionSafety);
            vf.setName(valueFunctionName);
            return vf;
        }
        if (valueFunctionType.equalsIgnoreCase("LookupTableZipped"))
        {
            System.out.println("USING LOOKUP TABLE VALUE FUNCTION.");
            ValueFunctionHashMap vf = new ValueFunctionHashMapZipped(sWorld, valueFunctionSafety);
            vf.setName(valueFunctionName);
            return vf;
        }
        
        throw new RuntimeException("\nvalueFunctionType must be either 'Perceptron', 'ResidualAlgorithm', 'LookupTable' or 'ResidualAlgorithmLinear'.  It was '"+valueFunctionType+"'\n");
    }
    
    private void trainAgent(Agent reader, ValueFunction valueFunction, ValueFunction tempNewValueFunction, int trial, int numStates)
    {
        if (rlAlgorithm.equalsIgnoreCase("valueIteration"))
        {
            long timerStart = System.currentTimeMillis();
            reader.performValueIteration(tempNewValueFunction, valueFunction);
            long timerCease = System.currentTimeMillis();
            displayTimeCalculations(timerStart, timerCease, trial, numStates);
        }
        else
        if (rlAlgorithm.equalsIgnoreCase("TD(lambda)"))
        {
            throw new RuntimeException("TD(lambda) is not implemented.");
        }
        else
        if (rlAlgorithm.equalsIgnoreCase("valueIterationTrajectorySample"))
        {
            System.out.println("Trajectory " + trial);
            reader.performValueIterationTrajectorySample(tempNewValueFunction, valueFunction);
        }
        else
        {
            System.out.println("Try:  valueIteration valueIterationTrajectorySample");
            throw new RuntimeException("rlAlgorithm unrecognized:  "+rlAlgorithm);
        }
    }
    
    private  void saveValueFunction(ValueFunction valueFunction, int trial)
    {
        if (saveNetworkEvery > 0 && ((trial % saveNetworkEvery) == 0))
        {
            if (!valueFunctionToSave.equals(""))
            {
                saveValueFunction(valueFunction, valueFunctionToSave+"-"+trial);
            }
        }
    }
    
    private  void dumpValueFunction(ValueFunction valueFunction, SentenceWorld sWorld)
    {
        Iterator stateIterator = sWorld.stateIterator();
        System.out.println("Value Function Dump");
        while (stateIterator.hasNext())
        {
            ReadingState rs = (ReadingState) stateIterator.next();
            System.out.println(rs + " = " + valueFunction.getValue(rs));
        }
    }
    
    private boolean receivedEndRunMessage()
    {
        File f = new File("endrun.msg");
        if (f.exists())
        {
            System.out.println("File endrun.msg exists, stopping run.");
            return true;
        }
        return false;
    }
    
    private String showVector(double [] x)
    {
        StringBuffer result = new StringBuffer();
        result.append("[");
        for (int i = 0; i < x.length; i++)
        {
            result.append(x[i] + " ");
        }
        result.append("]");
        return result.toString();
    }
    
    private ValueFunction loadValueFunction(String filename)
    {
        System.out.println("Loading pre-existing value function.");
        try
        {
            ObjectInputStream in = new ObjectInputStream(new GZIPInputStream(new FileInputStream(filename+".gz")));
            ValueFunction vf = null;
            vf = (ValueFunction) in.readObject();
            System.out.println("Value function of type "+vf.getClass()+" loaded from " + filename);
            if (vf instanceof ValueFunctionResidualAlgorithmLinear)
            {
                System.out.println("ValueFunction is linear network!  Re-setting parameters in case changed.");
                System.out.println("  learning rate: " + this.learningRate);
                ((ValueFunctionResidualAlgorithmLinear) vf).setLearningRate(this.learningRate);
                System.out.println("  residual weighting: " + this.residualWeighting);
                ((ValueFunctionResidualAlgorithmLinear) vf).setResidualWeighting(this.residualWeighting);
            }
            if (vf instanceof ValueFunctionResidualAlgorithmPerceptron)
            {
                System.out.println("ValueFunction is perceptron residual algorithm network!  Re-setting parameters in case changed.");
                System.out.println("  learning rate: " + this.learningRate);
                ((ValueFunctionResidualAlgorithmPerceptron) vf).setLearningRate(this.learningRate);
                System.out.println("  residual weighting: " + this.residualWeighting);
                ((ValueFunctionResidualAlgorithmPerceptron) vf).setResidualWeighting(this.residualWeighting);
            }
            if (vf instanceof ValueFunctionPerceptron)
            {
                System.out.println("ValueFunction is perceptron!  Re-setting parameters in case changed.");
                System.out.println("  learning rate: " + this.learningRate);
                ((ValueFunctionPerceptron) vf).setLearningRate(this.learningRate);
            }
            if (wrapSerialValueFunctionWithParallelAdapter)
            {
                System.out.println("Wrapping loaded value function in Parallel-to->Serial Adapter.");
                vf = new ParallelToSerialVFAdapter(vf);
            }
            return vf;
        }
        catch (Exception e)
        {
            System.out.println("Reading value function from file "+filename+" failed.\n");
            e.printStackTrace();
            throw new RuntimeException("Value Function loading failed.");
        }
    }
    
    protected Corpus testCorpus;
    protected Corpus trainCorpus;
    
    private void loadCorpus(Corpus corpus, String filename)
    {
        corpus = new Corpus();
        try
        {        
            BufferedReader reader = new BufferedReader(new FileReader(filename));
            while (reader.ready())
            {
                String sentence = reader.readLine();
                String times = reader.readLine();
                String predictabilities = reader.readLine();
                
                corpus.addSentence(sentence, "", times, predictabilities);
            }
        }
        catch (Exception e)
        {
            e.printStackTrace();
        }
    }
    
    private void displayTimeCalculations(long timerStart, long timerCease, int j, int numStates)
    {
        System.out.println("Iteration "+j+" took "+(timerCease-timerStart)+" ms");
        System.out.print("Rate: "+((double)(timerCease-timerStart))/(((double)(numStates)))+" ms/state ");
        
        double minutes = ((timerCease-timerStart) * (double) (totalIterations-j) / (double) 1000 / (double) 60);
        double hours = minutes / 60d;
        System.out.print("Time left: " + minutes + " minutes, which is "+hours+" hours");
    }
    
    
    private void displayTimeCalculationsForTrials(long timerStart, long timerCease, int runsPerTrial, int trial)
    {
        double timePerBlock = (double) timerCease - (double) timerStart;
        double blocksLeft = (double) totalIterations - (double) trial;
        double timeLeft = blocksLeft * timePerBlock;
        
        double minutes = timeLeft / 1000d / 60d;
        double hours = minutes / 60d;
        System.out.println(timePerBlock + "ms per block. Time left: " + minutes + " minutes, which is "+hours+" hours");
    }
    
    private void saveValueFunction(ValueFunction valueFunction, String filename)
    {
        System.out.println("Saving value function.");
        try
        {
            GZIPOutputStream zipout = new GZIPOutputStream(new FileOutputStream(filename+".gz"));
            ObjectOutputStream out = new ObjectOutputStream(zipout);
            out.writeObject(valueFunction);
            out.close();
            System.out.println("Value Function object of type " + valueFunction.getClass() + " has been written to file.\n");
        }
        catch (Exception e)
        {
            System.out.println("Writing value function object to file failed.\n");
            e.printStackTrace();
        }
        System.out.println("Saving value function. [done]");
    }
    
    private void showProperties()
    {
        System.out.println("ReadingMain VERSION " + VERSION);
        System.out.println("------------ properties ----------------------");
        System.out.println("Random Seed: '" + randomSeed +"'");
        System.out.println("totalIterations = '" + totalIterations +"'");
        System.out.println("startState = " + startState);
        System.out.println("SENTENCE ENVIRONMENT");
        System.out.println(" sentence =  '" + sentence +"'");
        System.out.println(" pred  =  '" + predictabilities +"'");
        System.out.println(" aid =  '" + aidTimes +"'");
        System.out.println(" testSentence =  '" + testSentence +"'");
        System.out.println(" testPred  =  '" + testPredictabilities +"'");
        System.out.println(" testAid =  '" + testAidTimes +"'");
        System.out.println(" saccPrgTime = '" + saccPrgTime +"'");
        System.out.println(" saccadicError = '" + saccadicError +"'");
        System.out.println(" saccadicErrorSpread = '" + saccadicErrorSpread +"'");
        System.out.println(" saccadicErrorGaussian = '" + saccadicErrorGaussian +"'");
        System.out.println(" acuityLinearPenalize = '" + acuityLinearPenalize + "'");
        System.out.println(" acuityLinearPenaltySlope = '" + acuityLinearPenaltySlope + "'");
        System.out.println(" saccadeTimePenalty = " + saccadeTimePenalty);
        System.out.println(" attention = " + attention);
        System.out.println(" attentionGradient: " );
        System.out.println("PARALLEL_UNIFORM_ATTENTION = " + PARALLEL_UNIFORM_ATTENTION);
        System.out.println("PARALLEL_LINEAR_ATTENTION_GRADIENT = " + PARALLEL_LINEAR_ATTENTION_GRADIENT);
        System.out.println("PARALLEL_DIVIDED_ATTENTION_GRADIENT = " + PARALLEL_DIVIDED_ATTENTION_GRADIENT);
        
        System.out.println(" acuityLimit = " + acuityLimit);
        System.out.println("FUNCTION APPROXIMATORS");
        System.out.println(" learningRate =  '" + learningRate +"'");
        System.out.println(" momentum =  '" + momentum+"'");
        System.out.println(" hiddenLayerSize =  '" + hiddenLayerSize +"'");
        System.out.println(" incrementalUpdate =  '" + incrementalUpdate +"'");
        System.out.println(" netOutputRange =  '" + netOutputRange +"'");
        System.out.println(" residualWeighting = '" + residualWeighting + "'");
        System.out.println(" stateVectorsCrossProduct = " + stateVectorsCrossProduct);
        System.out.println(" stateVectorsComplement = " + stateVectorsComplement);
        System.out.println(" residualAlgorithm = " + residualAlgorithm);
        System.out.println("REINFORCEMENT LEARNING");
        System.out.println(" rlAlgorithm = '" + rlAlgorithm +"'");
        System.out.println(" discountFactor =  '" + discountFactor +"'");
        System.out.println(" epsilon =  '" + epsilon +"'");
        System.out.println(" greed = '" + greed +"'");
        System.out.println(" testTrainSameVf = '" + testTrainSameVf +"'");
        System.out.println("VALUE FUNCTION");
        System.out.println(" reachTerminalReinforcement =  '" + reachTerminalReinforcement +"'");
        System.out.println(" valueOfTerminalStates =  '" + valueOfTerminalStates +"'");
        System.out.println(" valueFunctionToLoad = '" + valueFunctionToLoad +"'");
        System.out.println(" valueFunctionToSave = '" + valueFunctionToSave +"'");
        System.out.println(" saveNetworkEvery = '" + saveNetworkEvery +"'");
        System.out.println(" valueFunctionType = " + valueFunctionType);
        System.out.println(" valueFunctionSafety = " + valueFunctionSafety);
        System.out.println(" valueFunctionStateBounds = " + valueFunctionStateBounds);
        System.out.println("# POLICY / REWARD SCHEDULE");
        System.out.println(" defaultReinforcement = " + defaultReinforcement +"");
        System.out.println(" identifyReinforcement = " + identifyReinforcement +"");
        System.out.println(" readingStateType = " + readingStateType +"");
        System.out.println(" valueOfNonStoredStates = "+valueOfNonStoredStates);
        System.out.println("bitPacker = '" + bitPacker +"'");
        System.out.println("stateActionCache = '" + stateActionCache +"'");
        System.out.println("# DEPRECATED");
        System.out.println(" loadWorldCache = '" + loadWorldCache +"'");
        System.out.println(" boundedBufferSize = '" + boundedBufferSize +"'");
        System.out.println("# COMPUTATION ENVIRONMENT");
        System.out.println(" agentType = " + agentType);
        System.out.println(" agentParallelNumThreads = " + agentParallelNumThreads);
        System.out.println(" distributedJavaParty = " + distributedJavaParty);
        System.out.println("parallelAttendableWords (if attention != serial) = " + parallelAttendableWords);
        System.out.println("centerOfAttention (if attention != serial) = " + centerOfAttention);
        System.out.println("ALLOW_SIMULTANEOUS_SACREQ_AND_ATTNEXT = " + ALLOW_SIMULTANEOUS_SACREQ_AND_ATTNEXT);
        System.out.println("ALLOW_SIMULTANEOUS_ATTNEXT_AND_IDENT = " + ALLOW_SIMULTANEOUS_ATTNEXT_AND_IDENT);
        System.out.println("DYNAMICALLY_LOAD_VALUE_FUNCTION_PATCHES = " + DYNAMICALLY_LOAD_VALUE_FUNCTION_PATCHES);
        System.out.println("valueFunctionName = " + valueFunctionName);
        System.out.println("handCraftedVectors = " + handCraftedVectors);
    }
    
    
    private void setStateValuesTo(ValueFunction valueFunction, List states, double value)
    {
        System.out.println("Making sure Terminal States have values Near 1.0");
        
        Iterator i = states.iterator();
        double maxAllowedDelta = 0.0001d;
        double maxDelta = maxAllowedDelta;
        while (maxDelta >= maxAllowedDelta)
        {
            maxDelta = 0d;
            while (i.hasNext())
            {
                State s = (State) i.next();
                valueFunction.setValue(s, value);
                double test = Math.abs(value - valueFunction.getValue(s));
                if (test > maxDelta)
                { maxDelta = test; }
            }
        }
        System.out.println("Making sure Terminal States have values Near 1.0 [Done]");
    }
    
    
    protected void loadProperties()
    {
        try
        {
            Properties properties = new Properties();
            properties.load(new FileInputStream("reading.prop"));
            if (properties.getProperty("trainCorpusFile") != null)
            {
                loadCorpus(trainCorpus, properties.getProperty("trainCorpusFile"));
            }
            if (properties.getProperty("testCorpusFile") != null)
            {
                loadCorpus(testCorpus, properties.getProperty("testCorpusFile"));
            }
            
            if (properties.getProperty("agentType") != null)
            {
                agentType = properties.getProperty("agentType").trim();
            }
            if (properties.getProperty("agentParallelNumThreads") != null)
            {
                agentParallelNumThreads = new Integer(properties.getProperty("agentParallelNumThreads").trim()).intValue();
                if (agentType.equalsIgnoreCase("serial"))
                { System.err.println("Warning: agentParallelNumThreads specified in setup file, but agentType = serial.  Execution will be serial."); }
            }
            if (properties.getProperty("randomSeed") != null)
            {
                randomSeed = new Long(properties.getProperty("randomSeed").trim()).longValue();
            }
            if (properties.getProperty("totalIterations") != null)
            {
                totalIterations = new Integer(properties.getProperty("totalIterations").trim()).intValue();
            }
            if (properties.getProperty("residualWeighting") != null)
            {
                residualWeighting = new Double(properties.getProperty("residualWeighting").trim()).doubleValue();
            }
            if (properties.getProperty("saccPrgTime") != null)
            {
                saccPrgTime = new Integer(properties.getProperty("saccPrgTime").trim()).intValue();
            }
            
            if (properties.getProperty("valueFunctionSafety") != null)
            {
                if ((properties.getProperty("valueFunctionSafety").trim()).equalsIgnoreCase("true"))
                {
                    valueFunctionSafety = true;
                }
                else
                {
                    valueFunctionSafety = false;
                }
            }
            if (properties.getProperty("valueFunctionStateBounds") != null)
            {
                if ((properties.getProperty("valueFunctionStateBounds").trim()).equalsIgnoreCase("true"))
                {
                    valueFunctionStateBounds = true;
                }
                else
                {
                    valueFunctionStateBounds = false;
                }
            }
            
            
            if (properties.getProperty("ALLOW_SIMULTANEOUS_SACREQ_AND_ATTNEXT") != null)
            {
                if ((properties.getProperty("ALLOW_SIMULTANEOUS_SACREQ_AND_ATTNEXT").trim()).equalsIgnoreCase("true"))
                {
                    ALLOW_SIMULTANEOUS_SACREQ_AND_ATTNEXT = true;
                }
                else
                {
                    ALLOW_SIMULTANEOUS_SACREQ_AND_ATTNEXT = false;
                }
            }
            if (properties.getProperty("ALLOW_SIMULTANEOUS_ATTNEXT_AND_IDENT") != null)
            {
                if ((properties.getProperty("ALLOW_SIMULTANEOUS_ATTNEXT_AND_IDENT").trim()).equalsIgnoreCase("true"))
                {
                    ALLOW_SIMULTANEOUS_ATTNEXT_AND_IDENT = true;
                }
                else
                {
                    ALLOW_SIMULTANEOUS_ATTNEXT_AND_IDENT = false;
                }
            }
            
            if (properties.getProperty("residualAlgorithm") != null)
            {
                if ((properties.getProperty("residualAlgorithm").trim()).equalsIgnoreCase("true"))
                {
                    residualAlgorithm = true;
                }
                else
                {
                    residualAlgorithm = false;
                }
            }
            
            if (properties.getProperty("testAgentEvery") != null)
            {
                testAgentEvery = new Integer(properties.getProperty("testAgentEvery")).intValue();
            }
            
            if (properties.getProperty("stateVectorsComplement") != null)
            {
                if ((properties.getProperty("stateVectorsComplement").trim()).equalsIgnoreCase("true"))
                {
                    stateVectorsComplement = true;
                }
                else
                {
                    stateVectorsComplement = false;
                }
            }
            if (properties.getProperty("stateVectorsCrossProduct") != null)
            {
                if ((properties.getProperty("stateVectorsCrossProduct").trim()).equalsIgnoreCase("true"))
                {
                    stateVectorsCrossProduct = true;
                }
                else
                {
                    stateVectorsCrossProduct = false;
                }
            }
            
            if (properties.getProperty("incrementalUpdate") != null)
            {
                if ((properties.getProperty("incrementalUpdate").trim()).equalsIgnoreCase("true"))
                {
                    incrementalUpdate = true;
                }
                else
                {
                    incrementalUpdate = false;
                }
            }
            
            if (properties.getProperty("hiddenLayerSize") != null)
            {
                hiddenLayerSize = new Integer(properties.getProperty("hiddenLayerSize").trim()).intValue();
            }
            if (properties.getProperty("acuityLinearPenaltySlope") != null)
            {
                acuityLinearPenaltySlope = new Integer(properties.getProperty("acuityLinearPenaltySlope").trim()).intValue();
            }
            if (properties.getProperty("acuityLimit") != null)
            {
                acuityLimit = new Integer(properties.getProperty("acuityLimit").trim()).intValue();
            }
            if (properties.getProperty("saveNetworkEvery") != null)
            {
                saveNetworkEvery = new Integer(properties.getProperty("saveNetworkEvery").trim()).intValue();
            }
            if (properties.getProperty("boundedBufferSize") != null)
            {
                boundedBufferSize = new Integer(properties.getProperty("boundedBufferSize").trim()).intValue();
            }
            if (properties.getProperty("testIterations") != null)
            {
                testIterations = new Integer(properties.getProperty("testIterations").trim()).intValue();
            }
            if (properties.getProperty("parallelAttendableWords") != null)
            {
                parallelAttendableWords = new Integer(properties.getProperty("parallelAttendableWords").trim()).intValue();
            }
            if (properties.getProperty("centerOfAttention") != null)
            {
                centerOfAttention = new Integer(properties.getProperty("centerOfAttention").trim()).intValue();
            }
            if (properties.getProperty("saccadeTimePenalty") != null)
            {
                saccadeTimePenalty = new Integer(properties.getProperty("saccadeTimePenalty").trim()).intValue();
            }
            if (properties.getProperty("netOutputRange") != null)
            {
                netOutputRange = new Double(properties.getProperty("netOutputRange").trim()).doubleValue();
            }
            if (properties.getProperty("momentum") != null)
            {
                momentum = new Double(properties.getProperty("momentum").trim()).doubleValue();
            }
            if (properties.getProperty("reachTerminalReinforcement") != null)
            {
                reachTerminalReinforcement = new Double(properties.getProperty("reachTerminalReinforcement").trim()).doubleValue();
            }
            if (properties.getProperty("valueOfNonStoredStates") != null)
            {
                valueOfNonStoredStates = new Double(properties.getProperty("valueOfNonStoredStates").trim()).doubleValue();
            }
            if (properties.getProperty("saccadicErrorSpread") != null)
            {
                saccadicErrorSpread = new Double(properties.getProperty("saccadicErrorSpread").trim()).doubleValue();
            }
            if (properties.getProperty("greed") != null)
            {
                greed = new Double(properties.getProperty("greed").trim()).doubleValue();
            }
            if (properties.getProperty("valueOfTerminalStates") != null)
            {
                valueOfTerminalStates = new Double(properties.getProperty("valueOfTerminalStates")).doubleValue();
            }
            if (properties.getProperty("defaultReinforcement") != null)
            {
                defaultReinforcement = new Double(properties.getProperty("defaultReinforcement").trim()).doubleValue();
            }
            if (properties.getProperty("identifyReinforcement") != null)
            {
                identifyReinforcement = new Double(properties.getProperty("identifyReinforcement").trim()).doubleValue();
            }
            if (properties.getProperty("discountFactor") != null)
            {
                discountFactor = new Double(properties.getProperty("discountFactor").trim()).doubleValue();
            }
            if (properties.getProperty("epsilon") != null)
            {
                epsilon = new Double(properties.getProperty("epsilon").trim()).doubleValue();
            }
            if (properties.getProperty("learningRate") != null)
            {
                learningRate = new Double(properties.getProperty("learningRate").trim()).doubleValue();
            }
            if (properties.getProperty("valueFunctionType") != null)
            {
                valueFunctionType = properties.getProperty("valueFunctionType").trim();
            }
            if (properties.getProperty("registerWithBitMapper") != null)
            {
                if ((properties.getProperty("registerWithBitMapper").trim()).equalsIgnoreCase("true"))
                {
                    throw new RuntimeException("\nRegisterWithBitMapper is no longer used; state vectors are now provided by the state objects themselves rather than an external utility class.");
                }
            }
            
            if (properties.getProperty("distributedJavaParty") != null)
            {
                if (properties.getProperty("distributedJavaParty").trim().equalsIgnoreCase("true"))
                {
                    distributedJavaParty = true;
                }
                else
                {
                    distributedJavaParty = false;
                }
            }
            if (properties.getProperty("saccadicErrorGaussian") != null)
            {
                if (properties.getProperty("saccadicErrorGaussian").trim().equalsIgnoreCase("true"))
                {
                    saccadicErrorGaussian = true;
                }
                else
                {
                    saccadicErrorGaussian = false;
                }
            }
            if (properties.getProperty("handCraftedVectors") != null)
            {
                if (properties.getProperty("handCraftedVectors").trim().equalsIgnoreCase("true"))
                {
                    handCraftedVectors = true;
                    System.out.println("@@@@@@@@@@@ WARNING @@@@@@@@@@@@");
                    System.out.println("Hand Crafted Vectors are likely to produce bizarre results.");
                    System.out.println("@@@@@ WARNING END @@@@@@");
                }
                else
                {
                    handCraftedVectors = false;
                }
            }
            if (properties.getProperty("testTrainSameVf") != null)
            {
                if (properties.getProperty("testTrainSameVf").trim().equalsIgnoreCase("true"))
                {
                    testTrainSameVf = true;
                    if (this.valueFunctionType.equalsIgnoreCase("LookupTable"))
                    {
                        System.out.println("@@@@@ WARNING @@@@@@");
                        System.out.println("Usually,  with lookup tables, you want to testAndTrainSameVf=false, to proceed on ");
                        System.out.println("different value functions during learning (swap)");
                        System.out.println("@@@@@ WARNING END @@@@@@");
                    }
                }
                else
                {
                    testTrainSameVf = false;
                    if (!this.valueFunctionType.equalsIgnoreCase("LookupTable"))
                    {
                        System.out.println("@@@@@ WARNING @@@@@@");
                        System.out.println("Usually, with function approximators, you want to testAndTrainSameVf=true, i.e. on\n" +
                                "the SAME value functions during learning (swap)");
                        System.out.println("@@@@@ WARNING END @@@@@@");
                    }
                }
            }
            if (properties.getProperty("bitPacker") != null)
            {
                if (properties.getProperty("bitPacker").trim().equalsIgnoreCase("true"))
                {
                    bitPacker = true;
                }
                else
                {
                    bitPacker = false;
                }
            }
            if (properties.getProperty("saccadicError") != null)
            {
                if (properties.getProperty("saccadicError").trim().equalsIgnoreCase("true"))
                {
                    saccadicError = true;
                }
                else
                {
                    saccadicError = false;
                }
            }
            
            if (properties.getProperty("attention") != null)
            {
                attention = properties.getProperty("attention");
            }
            
            if (properties.getProperty("attentionGradient") != null)
            {
                if (properties.getProperty("attentionGradient").trim().equalsIgnoreCase("uniform"))
                {
                    PARALLEL_UNIFORM_ATTENTION = true;
                    PARALLEL_LINEAR_ATTENTION_GRADIENT = false;
                    PARALLEL_DIVIDED_ATTENTION_GRADIENT = false;
                }
                else
                    if (properties.getProperty("attentionGradient").trim().equalsIgnoreCase("linear"))
                    {
                    PARALLEL_UNIFORM_ATTENTION = false;
                    PARALLEL_LINEAR_ATTENTION_GRADIENT = true;
                    PARALLEL_DIVIDED_ATTENTION_GRADIENT = false;
                    }
                    else
                        if (properties.getProperty("attentionGradient").trim().equalsIgnoreCase("divided"))
                        {
                    PARALLEL_UNIFORM_ATTENTION = false;
                    PARALLEL_LINEAR_ATTENTION_GRADIENT = false;
                    PARALLEL_DIVIDED_ATTENTION_GRADIENT = true;
                        }
                        else
                        {
                    throw new RuntimeException("Unrecognized attentionGradient parameter: " + properties.getProperty("attentionGradient")+".  Try unform, linear or divided.");
                        }
                if (attention.equals("serial"))
                {
                    System.out.println("@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@");
                    System.out.println("WARNING: In serial mode, it is meaningless to set the attention gradient.");
                    System.out.println("@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@");
                }
            }
            else
            {
                if (!attention.equals("serial"))
                {
                    System.out.println("@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@");
                    System.out.println("WARNING: In nonserial mode, you should probably set an attention gradient.");
                    System.out.println("@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@");
                }
            }


            if (properties.getProperty("updateOnlyWhenGreedy") != null)
            {
                if (properties.getProperty("updateOnlyWhenGreedy").trim().equalsIgnoreCase("true"))
                {
                    updateOnlyWhenGreedy = true;
                }
                else
                {
                    updateOnlyWhenGreedy = false;
                }
            }
            if (properties.getProperty("acuityLinearPenalize") != null)
            {
                if (properties.getProperty("acuityLinearPenalize").trim().equalsIgnoreCase("true"))
                {
                    acuityLinearPenalize = true;
                }
                else
                {
                    acuityLinearPenalize = false;
                }
            }
            if (properties.getProperty("stateActionCache") != null)
            {
                if (properties.getProperty("stateActionCache").trim().equalsIgnoreCase("true"))
                {
                    stateActionCache = true;
                }
                else
                {
                    stateActionCache = false;
                }
            }
            
            
            if (properties.getProperty("sentence") != null)
            {
                predictabilities = properties.getProperty("pred").trim();
                aidTimes = properties.getProperty("aid").trim();
                sentence = properties.getProperty("sentence").trim();
                if (properties.getProperty("lengths") != null)
                {
                    wordLengths = properties.getProperty("lengths").trim();
                    this.wordLengthsOverridden = true;
                }
            }
            if (properties.getProperty("testSentence") != null)
            {
                testPredictabilities = properties.getProperty("testPred").trim();
                testAidTimes = properties.getProperty("testAid").trim();
                testSentence = properties.getProperty("testSentence").trim();
                if (properties.getProperty("testLengths") != null)
                {
                    testWordLengths = properties.getProperty("testLengths").trim();
                    this.testWordLengthsOverridden = true;
                }
            }
            if (properties.getProperty("valueFunctionToLoad") != null)
            {
                valueFunctionToLoad = properties.getProperty("valueFunctionToLoad").trim();
            }
            if (properties.getProperty("wrapSerialValueFunctionWithParallelAdapter") != null)
            {
                if (properties.getProperty("wrapSerialValueFunctionWithParallelAdapter").trim().equalsIgnoreCase("true"))
                {
                    wrapSerialValueFunctionWithParallelAdapter = true;
                }
                else
                {
                    wrapSerialValueFunctionWithParallelAdapter = false;
                }
            }
            if (properties.getProperty("valueFunctionToSave") != null)
            {
                valueFunctionToSave = properties.getProperty("valueFunctionToSave").trim();
            }
            if (properties.getProperty("startState") != null)
            {
                startState = properties.getProperty("startState");
            }
            if (properties.getProperty("loadWorldCache") != null)
            {
                loadWorldCache = properties.getProperty("loadWorldCache").trim();
            }
            if (properties.getProperty("readingStateType") != null)
            {
                readingStateType = properties.getProperty("readingStateType").trim();
                if (!readingStateType.equalsIgnoreCase("Absolute") && !readingStateType.equalsIgnoreCase("Relative"))
                {
                    throw new RuntimeException("Reading State Type must either be relative or absolute.");
                }
            }
            if (properties.getProperty("rlAlgorithm") != null)
            {
                rlAlgorithm = properties.getProperty("rlAlgorithm").trim();
            }
            if (properties.getProperty("DYNAMICALLY_LOAD_VALUE_FUNCTION_PATCHES") != null)
            {
                if (properties.getProperty("DYNAMICALLY_LOAD_VALUE_FUNCTION_PATCHES").equalsIgnoreCase("true"))
                {
                    DYNAMICALLY_LOAD_VALUE_FUNCTION_PATCHES = true;
                }
                else
                {
                    DYNAMICALLY_LOAD_VALUE_FUNCTION_PATCHES = false;
                }
            }
            if (properties.getProperty("valueFunctionName") != null)
            {
                valueFunctionName = properties.getProperty("valueFunctionName").trim();
            }
            
        }
        catch (Exception e)
        {
            System.out.println("Problem processing 'reading.prop' properties file, stopping.");
            e.printStackTrace();
            System.exit(-1);
        }
    }
    
}
