package net.pakl.rl;

import java.util.LinkedList;

/** This class is a synchronized queue (FIFO) data structure.
 * It is often used to transfer objects between threads.
 *
 * NB: Do NOT attempt code like:
 *     if(!queue.isEmpty())
 *         foo = queue.get();
 * It is not thread-safe, even though isEmpty() and get() are synchronized.
 * It is possible for a thread to sneak in after the call to isEmpty() and before the call to get().
 * If you must do this, use something like:
 *     synchronized(queue)
 *     {
 *         if(!queue.isEmpty())
 *             foo.queue.get();
 *     }
 * The same applies to using size() and get().
 *
 */
public class Queue
{
    private LinkedList list;
    
    public Queue()
    {
        list = new LinkedList();
    }
    
    /** place an object onto the end of the queue
     */
    synchronized public void put(Object v)
    {
        list.addFirst(v);
        notify();
    }
    
    /** get an object off of the front of the queue
     */
    synchronized public Object get(long timeout)
    {
        try
        {
            //NB: if
            if(list.isEmpty())
            {
                wait(timeout);
            }
        }
        catch(InterruptedException e)
        {
            System.err.println("Interrupted, but not by a notify() call.");
            return null;
        }
        
        if(list.isEmpty())
        {
            return null;
        }
        else
        {
            return list.removeLast();
        }
    }
    
    synchronized public Object get()
    {
        try
        {
            //NB: while
            while(list.isEmpty())
            {
                wait();
            }
        }
        catch(InterruptedException e)
        {
            System.err.println("Interrupted, but not by a notify() call.");
            return null;
        }
        
        return list.removeLast();
    }
    
    synchronized public boolean isEmpty()
    {
        return list.isEmpty();
    }
    
    synchronized public int size()
    {
        return list.size();
    }
}
