package net.pakl.tools;

import java.util.*;
import java.io.*;


// example input file
//
// channel 1 @ 0 = 100 Hz
// channel 1 @ 50 = 100 Hz
// channel 1 @ 100 = 200 Hz
// channel 2 @ 0 = 200 Hz

public class KlattScheduler
{
    HashMap<Integer,double[]> data = new HashMap<Integer,double[]>();
    HashMap<String,Integer> variableNamesToIndices = new HashMap<String,Integer>();
    
    public static int [] defaultFrame = new int[40];
    public KlattScheduler()
    {
        variableNamesToIndices.put("F0", 0);
        variableNamesToIndices.put("AV", 1);
        variableNamesToIndices.put("F1", 2);
        variableNamesToIndices.put("B1", 3);
        variableNamesToIndices.put("F2", 4);
        variableNamesToIndices.put("B2", 5);
        variableNamesToIndices.put("F3", 6);
        variableNamesToIndices.put("B3", 7);
        variableNamesToIndices.put("F4", 8);
        variableNamesToIndices.put("B4", 9);
        variableNamesToIndices.put("F5", 10);
        variableNamesToIndices.put("B5", 11);
        variableNamesToIndices.put("F6", 12);
        variableNamesToIndices.put("B6", 13);
        variableNamesToIndices.put("FNZ", 14);
        variableNamesToIndices.put("BNZ", 15);
        variableNamesToIndices.put("FNP", 16);
        variableNamesToIndices.put("BNP", 17);
        variableNamesToIndices.put("ASP", 18);
        variableNamesToIndices.put("KOPEN", 19);
        variableNamesToIndices.put("ATURB", 20);
        variableNamesToIndices.put("TILT", 21);
        variableNamesToIndices.put("AF", 22);
        variableNamesToIndices.put("SKEW", 23);
        variableNamesToIndices.put("A1", 24);
        variableNamesToIndices.put("B1P", 25);
        variableNamesToIndices.put("A2", 26);
        variableNamesToIndices.put("B2P", 27);
        variableNamesToIndices.put("A3", 28);
        variableNamesToIndices.put("B3P", 29);
        variableNamesToIndices.put("A4", 30);
        variableNamesToIndices.put("B5P", 31);
        variableNamesToIndices.put("A5", 32);
        variableNamesToIndices.put("B5P", 33);
        variableNamesToIndices.put("A6", 34);
        variableNamesToIndices.put("B6P", 35);
        variableNamesToIndices.put("ANP", 36);
        variableNamesToIndices.put("AB", 37);
        variableNamesToIndices.put("AVP", 38);
        variableNamesToIndices.put("GAIN", 39);
    }
    
    String filename;
    int numChannels = 5;
    int granularity = 1;
    int timesteps = 200;
    double [][] frequencies; 
    double [][] amplitudes; 
    
    
    
    public void process(String filename) throws Exception
    {
        load(filename);
        convertHashMapDataToArrays();
        
        for (int columnNumber = 0; columnNumber < 40; columnNumber++)
        {
            double [] column = extractColumn(arraydata, columnNumber);
            interpolate(column);
            writeColumn(column, arraydata, columnNumber);
        }
        for (int t = 1; t < timesteps; t++)
        {
            setDefaults(arraydata[t]);
        }
        
        save(filename+".freq");
    }
    
    private boolean isSeparator(String s)
    {
        if (s.equals("|")) return true;
        if (s.equals(";")) return true;
        if (s.equals(",")) return true;
        if (s.equals("&")) return true;
        return false;
    }
    private double [] extractColumn(double [][] array, int columnNumber)
    {
        double [] result = new double[array.length];
        for (int i = 0; i < array.length; i++)
        {
            result[i] = array[i][columnNumber];
        }
        return result;
    }
    
    private void writeColumn(double [] column, double [][] array, int columnNumber)
    {
        for (int i = 0; i < array.length; i++)
        {
            array[i][columnNumber] = column[i];
        }
    }
    
    
    public void save(String filename) throws Exception
    {
        BufferedWriter b = new BufferedWriter(new FileWriter(filename));
        for (int t = 1; t < timesteps; t++)
        {
                b.write(implode(arraydata[t])+"\n");
        }
        b.close();
    }
        
    public int findNonDefault(double [] values, int startingFrom)
    {
        for (int i = startingFrom; i < values.length; i++)
        {
            if (values[i] >= 0)
            {
                return i;
            }
        }
        return values.length-1;
    }
    
    private void copyRowFromTo(double [] a, double [] b)
    {
        for (int i = 0; i < a.length; i++)
        {
            b[i] = a[i];
        }
    }
    
    double [][] arraydata;
    private void convertHashMapDataToArrays()
    {
        arraydata = new double[timesteps][40];

        for (int t = 0; t < timesteps; t++)
        {
            if (data.get(t) != null)
            {
                copyRowFromTo(data.get(t), arraydata[t]);
            }
            else
            {
                setDefaultFlags(arraydata[t]);
            }
        }
        
    }
            
    
    /** Interpolates between nonzero values */
    private void interpolate(double [] values)
    {
        int startTime = 0, endTime = 0;
        while (endTime < (timesteps-1))
        {
            startTime = findNonDefault(values, endTime);
            endTime = findNonDefault(values, startTime+1);

            if ((endTime == 0) || (startTime == (timesteps-1)) || ((endTime-startTime)<=0))
            {
                break;
            }
            interpolate(values, startTime, endTime);
        }
    }
    
    
    
    private void interpolate(double values[], int start, int end)
    {
        double slope = (values[end] - values[start]) / (double)(end-start);
        int steps = 0;
        for (int i = start+1; i <= (end-1); i++)
        {
            steps++;
            values[i] = values[start] + slope * steps;
        }
        
    }
    
    private void load(String filename) throws Exception
    {
        int currentTime = 1;
        Scanner s = new Scanner(new File(filename));
        if (s.next().equals("timesteps"))
        {
            timesteps = s.nextInt();
        }
        
        frequencies = new double[numChannels][granularity * timesteps];
        amplitudes = new double[numChannels][granularity * timesteps];
        
        while (s.hasNext())
        {
            String nextToken = s.next();
            while (isSeparator(nextToken)) nextToken = s.next();
            if (nextToken.equals("t")) 
            { 
                s.next(); // equals
                currentTime = s.nextInt();
                // System.out.println("Current time is now " + currentTime);
            }
            else
            {
                String variableName = nextToken;
                // System.out.println("Variable = " + variableName);
                s.next(); // equals
                int value = s.nextInt();
                setVariable(currentTime, variableName, value);
            }
        }
    }
    
    
    public void setVariable(int time, String variableName, int value)
    {
        double [] row = null;
        if (data.get(time)==null)
        {
            row = new double[40];
            setDefaultFlags(row);
        }
        else
        {
            row = data.get(time);
        }
        if (variableNamesToIndices.get(variableName)==null)
        {
            throw new RuntimeException("\nUnknown variable "+ variableName + " was used.");
        }
        row[variableNamesToIndices.get(variableName)] = value;
        data.put(time, row);
    }
    
    private void setDefaultFlags(double [] row)
    {
        for (int i = 0; i < row.length; i++)
        {
            row[i] = -1.0;
            // row[i] = 0;
        }
    }
    
    private void setDefaults(double [] row)
    {
        for (int i = 0; i < row.length; i++)
        {
            if (row[i] < 0)
            {
                row[i] = (double) defaultFrame[i];
            }
        }
    }

    public static void main(String args[]) throws Exception
    {
//        String dummyFrame =     "1000 70 497 0 739 0 2772 0 3364 0 4170 0 4000 0 0 0 200 40  0 40  0 20  0  0 53 44 79 70 52 95 44 56 34 80  0 80  0  0 27 0";
        String dummyFrame = "1000 70 250 0 1250 0 3000 0 3500 0 4170 0 4000 0 0 0 200 40 0 40 0 20 0 0 60 44 60 70 60 95 60 56 60 80 0 80 0 0 60 0 ";
        defaultFrame = explodeToInts(dummyFrame, " ");
        if (args.length == 0)
        {
            System.err.println("Pass filename as first argument");
            System.exit(1);
        }
        (new KlattScheduler()).process(args[0]);
    }
    
    public static int [] explodeToInts(String s, String delim)
    {
        StringTokenizer t = new StringTokenizer(s, delim);
        int [] result = new int[t.countTokens()];

        Scanner sc = new Scanner(s);
        int i = 0;
        while (sc.hasNextInt())
        {
            result[i] = sc.nextInt();
            i++;
        }
        return result;
    }
    
    public static String implode(double [] a)
    {
        String result = "";
        for (int i = 0; i < a.length; i++)
        {
            result = result + ((int)a[i]) + " ";
        }
        return result;
    }
}
