
package net.pakl.neuralnet;

import java.io.*;
import java.util.*;

public class VoiceTrainer
{
    public static String CONFIGURATION_FILE = "voicetrainer.prop";
    public static final String VERSION_NUMBER = "3.3(randomSeed/dont-recenter-inputs-only-feedback/feedforward-in-drive)";	// replicate input vectors at startup, not on fly
    SimpleRecurrentNetAverager srna;
    int hiddenUnitsPerInputVector = 50;
    int stutterTimeSteps = 1;			// Each timestep should default be presented only once.
    int batchesPerTrial = 1;
    String inputList = "badra.txt";
    String outputCodeList = "";
    String probeList = "badra.txt";
    String srnaToLoad = "";
    String srnaToSave = "srna-out.obj";
    boolean enableBatchUpdate = false;
    boolean clearContextUnits = false;
    boolean normalizeInputs = true;
    boolean divideByMaxInputs = false;
    int saveEvery = 0;
    public int predictionDelay = 1;		// v 1.10 editable
    int inputPatternWidth = 50;
    int outputPatternWidth = inputPatternWidth;
    int trials = 100;
    boolean restrictedFeedforward = false;
    int numInputCopies = 3;
    double preserveFactor = 0;
    double learningRate = 0.01;
    double momentumTerm = 0.01;
    long randomSeed = 123;
    double initialRandomWeightFactor = 0.01;        // small initial random weights range from -thisx0.5 to +thisx0.5
    
    boolean softClamping = false;
    double softClampingScalar = 0.5d;		// Amount by which to relatively weight the input stimuli compared to the networks own feedback.
    double inputScaleFactor = 1.0d;			// Amount by which to multiply/scale the net input (before the sigmoid).  Makes sigmoid more extreme.
    boolean clearSoftClampingFromInputs = true;
    
    boolean selectiveHardClamping = false;
    int hardClampStartUnit = 0;
    int hardClampCeaseUnit = 0;
    
    Random rng;
    
    public static void main(String args[]) throws Exception
    {
        VoiceTrainer vt = new VoiceTrainer();
        vt.loadProperties(CONFIGURATION_FILE);
        vt.run();
        vt.runProbes();
    }
    
    public void runProbes() throws Exception
    {
        System.out.println("runProbes()");
        List soundPatternsInput = readSoundPatterns(stringToList(probeList), true);
        List soundPatternsOutput = readSoundPatterns(stringToList(probeList), false);
        probe(probeList, soundPatternsInput, soundPatternsOutput);
    }
    
    public void run() throws Exception
    {
        List soundPatternsInput = readSoundPatterns(stringToList(inputList), true);
        List soundPatternsOutput = readSoundPatterns(stringToList(inputList), false);
        Stopwatch stopwatch = new Stopwatch();
        for (int i = 0; i < trials; i++)
        {
            System.out.print("Training Trial " + (i+1) + " of " + trials + "\n");
            stopwatch.start();
            train(soundPatternsInput, soundPatternsOutput);
            stopwatch.stop();
            System.out.println(stopwatch.getResult());
            if (saveEvery > 0)
            {
                if ((i % saveEvery) == 0)
                {
                    (new ObjectOutputStream(new FileOutputStream(srnaToSave+"-"+i))).writeObject(srna);
                }
            }
            if (receivedEndRunMessage()) break;
        }
        if (!srnaToSave.equals(""))
        {
            (new ObjectOutputStream(new FileOutputStream(srnaToSave))).writeObject(srna);
        }
        recordInputsAndOutputs(soundPatternsInput);
    }
    
    private boolean receivedEndRunMessage()
    {
        File f = new File("endrun.msg");
        if (f.exists())
        {
            System.out.println("File endrun.msg exists, stopping run.");
            return true;
        }
        return false;
    }
    
    protected double[] replicateInputVector(double[] inputVector)
    {
        // All we have to do is present the input vector numInputCopies times;  the SRNA itself
        // will take care of creating the appropriate running averager effects.
        double [] result = new double[inputVector.length * numInputCopies];
        for (int i = 0; i < inputVector.length * numInputCopies; i++)
        {
            result[i] = inputVector[i % inputVector.length];
        }
        return result;
    }
    
    class Triplet
    {
        private Object a, b, c;
        public Triplet(Object a, Object b, Object c)
        {
            this.a = a;
            this.b = b;
            this.c = c;
        }
        public Object get1()
        { return a; }
        public Object get2()
        { return b; }
        public Object get3()
        { return c; }
    }
    
    public void train(List inputs, List outputs)
    {
        rng = new Random(randomSeed);
        ArrayList triplets = new ArrayList();
        for (int i = 0; i < inputs.size(); i++)
        {
            triplets.add(new Triplet(new Integer(i), inputs.get(i), outputs.get(i)));
        }
        Collections.shuffle(triplets, rng);
        
        for (int i = 0; i < triplets.size(); i++)
        {
            System.out.print("Training sound pattern " + ((Triplet)triplets.get(i)).get1() + " ");
            List inputPattern = (List)  ((Triplet)triplets.get(i)) .get2();
            List outputPattern = (List) ((Triplet)triplets.get(i)) .get3();
            
            int patternSize = ((double[]) inputPattern.get(0)).length;
            double [] previouslyGeneratedOutput = new double[patternSize];
            
            for (int batchNum = 0; batchNum < batchesPerTrial; batchNum++)
            {
                srna.clearSSEMeasure();
                if (clearContextUnits)
                { srna.clearContextUnits(); }
                if (clearSoftClampingFromInputs)
                { previouslyGeneratedOutput = new double[patternSize]; }
                srna.resetRunningAveragers();
                for (int j = 0; j < inputPattern.size() - predictionDelay; j++)
                {
                    double [] forward = (double[]) inputPattern.get(j);
                    double [] teacher = (double[]) outputPattern.get(j + predictionDelay);
                    
                    if (softClamping == false)
                    {
                        srna.feedforward(forward);
                    }
                    else
                    {
                        // softClampingScalar can range from 0 to 1, lower number meaning more feedback excitation.
                        double [] clampedInput = vmul(softClampingScalar, forward);
                        double [] feedbackExcitation = vmul(1-softClampingScalar, recenterOnZero(previouslyGeneratedOutput));
                        double [] softClampedInputPlusFeedback = vadd(clampedInput, feedbackExcitation);
                        double [] softClampedFeedforward = vmul(inputScaleFactor, softClampedInputPlusFeedback);
                        srna.feedforward(softClampedFeedforward);
                        
                        previouslyGeneratedOutput = srna.getOutputLayerActivity();
                    }
                    
                    srna.backpropogate(teacher);
                    srna.copyHiddenExcitationToContextUnits();
                }
                if (enableBatchUpdate)
                {
                    srna.batchUpdateWeights();
                }
            }
            System.out.println("sse = " + srna.getSumSquaredError());
        }
    }
    
    
    protected double [] recenterOnZero(double [] pattern)
    {
        double [] result = vmul(2.0d, vadd(-0.5d, pattern));
        return result;
    }
    
    protected double [] scaleDown(double [] forward)
    {
        return vmul(1-softClampingScalar, forward);
    }
    
    protected double [] vcomplement(double [] v)
    {
        double [] result = new double[v.length];
        for (int i = 0; i < v.length; i++)
        {
            if (v[i] == 1.0d)
            {
                result[i] = 0.0d;
            }
            else
                if (v[i] == 0.0d)
                {
                result[i] = 1.0d;
                }
                else
                {
                throw new RuntimeException("To use soft clamping, the original input vector must be binary.  Clearly it is not, as v["+i+"] = " + v[i]);
                }
        }
        return result;
    }
    
    
    protected double [] vadd(double s, double [] v)
    {
        double [] result = new double[v.length];
        for (int i = 0; i < v.length; i++)
        {
            result[i] = s + v[i];
        }
        return result;
    }
    
    
    private int max(int a, int b)
    {
        if (a>b) return a;
        return b;
    }
    
    
    protected double [] vadd(double [] v1, double [] v2)
    {
        double [] result = new double[max(v1.length, v2.length)];
        int j = 0; int k = 0;
        for (int i = 0; i < max(v1.length, v2.length); i++)
        {
            result[i] = v1[j] + v2[k];
            j++; k++;
            if (j == v1.length)
            { j = 0; }		// loops around to beginning of vector
            if (k == v2.length)
            { k = 0; }		// loops around to beginning of vector
        }
        return result;
    }
    
    protected double [] vmul(double scalar, double [] vector)
    {
        double [] result = new double[vector.length];
        for (int i = 0; i < vector.length; i++)
        {
            result[i] = scalar * vector[i];
        }
        return result;
    }
    
    
    public void trainSequentially(List inputs, List outputs)
    {
        for (int i = 0; i < inputs.size(); i++)
        {
            System.out.print("Training sound pattern " + i + " ");
            List inputPattern = (List)inputs.get(i);
            List outputPattern = (List)outputs.get(i);
            srna.clearSSEMeasure();
            if (clearContextUnits)
            { srna.clearContextUnits(); }
            srna.resetRunningAveragers();
            for (int j = 0; j < inputPattern.size() - predictionDelay; j++)
            {
                double [] forward = (double[]) inputPattern.get(j);
                double [] backward = (double[]) outputPattern.get(j + predictionDelay);
                srna.feedforward(forward);
                srna.backpropogate(backward);
                srna.copyHiddenExcitationToContextUnits();
            }
            System.out.println("sse = " + srna.getSumSquaredError());
        }
        
    }
    
    public void probe(String inputNames, List soundPatternsInput, List soundPatternsOutput) throws Exception
    {
        System.out.println("probe()");
        for (int i = 0; i < soundPatternsInput.size(); i++)
        {
            List soundPatternInput = (List)soundPatternsInput.get(i);
            List soundPatternOutput = (List)soundPatternsOutput.get(i);
            BufferedWriter fullProbeOutputFile = new BufferedWriter( new FileWriter(stringToList(inputNames).get(i)+"-out-fullprobe.txt"));
            int fullProbe = soundPatternInput.size() - predictionDelay - 1;
            for (int driveUpTo = 0; driveUpTo < soundPatternInput.size() - predictionDelay; driveUpTo++)
            {
                BufferedWriter outputVectorFile = new BufferedWriter( new FileWriter(stringToList(inputNames).get(i)+"-out-drive-"+driveUpTo+".txt"));
                BufferedWriter inputVectorFile = new BufferedWriter( new FileWriter(stringToList(inputNames).get(i)+"-in-drive-"+driveUpTo+".txt"));
                if (clearContextUnits)
                { srna.clearContextUnits(); }
                srna.resetRunningAveragers();
                double [] previouslyGeneratedOutput = new double[((double[])soundPatternInput.get(0)).length];
                for (int j = 0; j < driveUpTo; j++)
                {
                    double [] forward = (double[]) soundPatternInput.get(j);
                    
                    
                    if (softClamping == false)
                    {
                        srna.feedforward(forward);
                        inputVectorFile.write(vectorToText(forward));
                    }
                    else
                    {
                        double [] clampedInput = vmul(softClampingScalar, forward);
                        double [] feedbackExcitation = vmul(1-softClampingScalar, recenterOnZero(previouslyGeneratedOutput));
                        double [] softClampedInputPlusFeedback = vadd(clampedInput, feedbackExcitation);
                        double [] softClampedFeedforward = vmul(inputScaleFactor, softClampedInputPlusFeedback);
                        inputVectorFile.write(vectorToText(softClampedFeedforward));
                        
                        srna.feedforward(softClampedFeedforward);
                        previouslyGeneratedOutput = srna.getOutputLayerActivity();
                    }
                    outputVectorFile.write(vectorToText((double[])srna.getOutputLayerActivity()));
                    if (driveUpTo == fullProbe)
                    {
                        fullProbeOutputFile.write(vectorToText((double[])srna.getOutputLayerActivity()));
                    }
                    srna.copyHiddenExcitationToContextUnits();
                }
                // Allow network output to feed back onto itself
                for (int j = driveUpTo; j < soundPatternInput.size() - predictionDelay; j++)
                {
                    // Since output of network is single, not replicated, we must replicate it.
                    srna.feedforward(replicateInputVector(srna.getOutputLayerActivity()));
                    inputVectorFile.write(vectorToText(replicateInputVector(srna.getOutputLayerActivity())));
                    outputVectorFile.write(vectorToText((double[])srna.getOutputLayerActivity()));
                    srna.copyHiddenExcitationToContextUnits();
                    if (driveUpTo == fullProbe)
                    {
                        fullProbeOutputFile.write(vectorToText((double[])srna.getOutputLayerActivity()));
                    }
                }
                outputVectorFile.close();
                inputVectorFile.close();
        }
        fullProbeOutputFile.close();
        }
    }
    
    public void recordInputsAndOutputs(List soundPatterns)
    {
        System.out.println("recordInputsAndOutputs()");
        try
        {
            for (int i = 0; i < soundPatterns.size(); i++)
            {
                System.out.print("Testing sound pattern " + i + " ");
                List soundPattern = (List)soundPatterns.get(i);
                srna.clearSSEMeasure();
                BufferedWriter inputVectorFile = new BufferedWriter(new FileWriter(stringToList(inputList).get(i)+"-in.txt"));
                BufferedWriter outputVectorFile = new BufferedWriter(new FileWriter(stringToList(inputList).get(i)+"-out.txt"));
                if (clearContextUnits)
                { srna.clearContextUnits(); }
                srna.resetRunningAveragers();
                double [] previouslyGeneratedOutput = new double[((double[]) soundPattern.get(0)).length];
                for (int j = 0; j < soundPattern.size() - predictionDelay; j++)
                {
                    double [] forward = (double[]) soundPattern.get(j);
                    
                    if (softClamping == false)
                    {
                        srna.feedforward(forward);
                        inputVectorFile.write(vectorToText(forward));
                    }
                    else
                    {
                        double [] clampedInput = vmul(softClampingScalar, forward);
                        double [] feedbackExcitation = vmul(1-softClampingScalar, recenterOnZero(previouslyGeneratedOutput));
                        double [] softClampedInputPlusFeedback = vadd(clampedInput, feedbackExcitation);
                        double [] softClampedFeedforward = vmul(inputScaleFactor, softClampedInputPlusFeedback);
                        srna.feedforward(softClampedFeedforward);
                        inputVectorFile.write(vectorToText(softClampedFeedforward));
                        previouslyGeneratedOutput = srna.getOutputLayerActivity();
                    }
                    outputVectorFile.write(vectorToText(srna.getOutputLayerActivity()));
                    srna.copyHiddenExcitationToContextUnits();
                }
                System.out.println("sse = " + srna.getSumSquaredError());
                inputVectorFile.close();
                outputVectorFile.close();
            }
        }
        catch (IOException ioe)
        {
            ioe.printStackTrace();
            System.out.println("Error -- could not write input and output vectors after training to disk.");
        }
    }
    
    protected String vectorToText(double [] a)
    {
        String result = "";
        for (int i = 0; i < a.length; i++)
        {
            result = result + (a[i]+" ");
        }
        return result + "\n";
    }
    
    
    public VoiceTrainer() throws Exception
    {
    }
    
    public void loadProperties(String propertiesFile) throws Exception
    {
        Properties p = new Properties();
        p.load(new FileInputStream(propertiesFile));
        if (p.getProperty("numInputCopies")  != null)   numInputCopies = new Integer(p.getProperty("numInputCopies")).intValue();
        if (numInputCopies < 1)
        { throw new RuntimeException("You must have at least 1 input copy to achieve learning."); }
        if (p.getProperty("batchesPerTrial") != null)   batchesPerTrial = new Integer(p.getProperty("batchesPerTrial")).intValue();
        if (p.getProperty("inputList")       != null)   inputList = p.getProperty("inputList");
        if (p.getProperty("outputCodeList")       != null)   outputCodeList = p.getProperty("outputCodeList");
        if (p.getProperty("clearContextUnits") != null)
        {
            if (p.getProperty("clearContextUnits").equalsIgnoreCase("true"))
            {
                clearContextUnits = true;
            }
            else
            {
                clearContextUnits = false;
            }
        }
        if (p.getProperty("probeList")       != null)         probeList = p.getProperty("probeList");
        if (p.getProperty("trials")          != null)       trials = new Integer(p.getProperty("trials")).intValue();
        if (p.getProperty("predictionDelay")          != null)       predictionDelay = new Integer(p.getProperty("predictionDelay")).intValue();
        if (p.getProperty("inputPatternWidth")          != null)
        {
            inputPatternWidth = new Integer(p.getProperty("inputPatternWidth")).intValue();
            outputPatternWidth = inputPatternWidth;
        }
        if (p.getProperty("stutterTimeSteps") != null)
        {
            stutterTimeSteps = new Integer(p.getProperty("stutterTimeSteps")).intValue();
        }
        if (p.getProperty("outputPatternWidth") != null)
        {
            if (this.getClass() == VoiceTrainer.class)
            {
                System.err.println("Warning -- for VoiceTrainer, outputPatternWidth should usually not "
                        + "be set -- it defaults to the inputPatternWidth.  Subclasses may override this, however.");
            }
            outputPatternWidth = new Integer(p.getProperty("outputPatternWidth")).intValue();;
        }
        if (p.getProperty("hiddenPerUnit") != null)	hiddenUnitsPerInputVector = new Integer(p.getProperty("hiddenPerUnit")).intValue();
        if (p.getProperty("randomSeed") != null)	randomSeed = new Integer(p.getProperty("randomSeed")).intValue();
        
        int numNeurons = numInputCopies*inputPatternWidth
                + numInputCopies*hiddenUnitsPerInputVector
                + outputPatternWidth;      // input, hidden and output. context neurons handled by srn.
        int [] layerBreaks = {  numInputCopies*inputPatternWidth,
                numInputCopies*inputPatternWidth + numInputCopies*hiddenUnitsPerInputVector,
                numInputCopies*inputPatternWidth + numInputCopies*hiddenUnitsPerInputVector+ outputPatternWidth};
                if (p.getProperty("learningRate")  != null)       learningRate = new Double(p.getProperty("learningRate")).doubleValue();
                if (p.getProperty("initialRandomWeightFactor") != null) initialRandomWeightFactor = new Double(p.getProperty("initialRandomWeightFactor")).doubleValue();
                
                if (p.getProperty("momentumTerm")  != null)       momentumTerm = new Double(p.getProperty("momentumTerm")).doubleValue();
                if (p.getProperty("softClampingScalar") != null)    softClampingScalar = new Double(p.getProperty("softClampingScalar")).doubleValue();
                if (p.getProperty("inputScaleFactor") != null)		inputScaleFactor = new Double(p.getProperty("inputScaleFactor")).doubleValue();
                
                if (p.getProperty("softClamping") != null)
                {
                    if (p.getProperty("softClamping").equals("false")) softClamping = false;
                    else softClamping = true;
                }
                if (p.getProperty("normalizeInputs") != null)
                {
                    if (p.getProperty("normalizeInputs").equalsIgnoreCase("true")) normalizeInputs = true;
                    else normalizeInputs = false;
                }
                if (p.getProperty("divideByMaxInputs") != null)
                {
                    if (p.getProperty("divideByMaxInputs").equalsIgnoreCase("true"))
                    {
                        divideByMaxInputs = true;
                        if (normalizeInputs == true)
                        {
                            System.out.println("NormalizeInputs was true, setting to false because divideByMaxInputs=true.");
                            normalizeInputs = false;
                        }
                    }
                    else divideByMaxInputs = false;
                }
                if (softClamping && normalizeInputs)
                {
                    throw new RuntimeException("To use true softclamping, the inputs must be boolean vectors.  Make normalizeInputs == false");
                }
                if (p.getProperty("clearSoftClampingFromInputs") != null)
                {
                    if (p.getProperty("clearSoftClampingFromInputs").equals("false")) clearSoftClampingFromInputs = false;
                    else clearSoftClampingFromInputs = true;
                }
                
                if (p.getProperty("preserveFactor")  != null)       preserveFactor = new Double(p.getProperty("preserveFactor")).doubleValue();
                
                
                
                if (p.getProperty("enableBatchUpdate") != null)
                {
                    if (p.getProperty("enableBatchUpdate").equalsIgnoreCase("true")) enableBatchUpdate = true;
                    else enableBatchUpdate = false;
                }
                if (p.getProperty("restrictedFeedforward") != null)
                {
                    if (p.getProperty("restrictedFeedforward").equalsIgnoreCase("true")) restrictedFeedforward = true;
                    else restrictedFeedforward = false;
                }
                if (p.getProperty("srnaToLoad") != null) srnaToLoad = p.getProperty("srnaToLoad");
                if (p.getProperty("srnaToSave") != null) srnaToSave = p.getProperty("srnaToSave");
                if (p.getProperty("saveEvery") != null) saveEvery = new Integer(p.getProperty("saveEvery")).intValue();
                
                if (srnaToLoad.equals(""))
                {
                    srna = new SimpleRecurrentNetAverager(randomSeed, numNeurons, learningRate, momentumTerm, layerBreaks, initialRandomWeightFactor);
                    srna.setBatchUpdate(enableBatchUpdate);
                }
                else
                {
                    srna = (SimpleRecurrentNetAverager) (new ObjectInputStream(new FileInputStream(srnaToLoad))).readObject();
                    System.out.println("Network loaded, setting parameters from current file.");
                    srna.setLearningRate(learningRate);
                    srna.setMomentumTerm(momentumTerm);
                }
                if (restrictedFeedforward)
                {
                    pruneInputToHidden(srna);
                }
                configureAveragerProperties(srna);
                showProperties();
    }
    
    protected void pruneInputToHidden(SimpleRecurrentNet srn)
    {
        for (int inputGroup = 0; inputGroup < numInputCopies; inputGroup++)
        {
            for (int hiddenGroup = 0; hiddenGroup < numInputCopies; hiddenGroup++)
            {
                if (inputGroup != hiddenGroup)
                {
                    for (int pre = inputGroup * inputPatternWidth; pre < (inputGroup+1)*inputPatternWidth; pre++)
                        for (int post = hiddenGroup * hiddenUnitsPerInputVector; post < (hiddenGroup+1) * hiddenUnitsPerInputVector; post++)
                        {
                        srn.disconnect(pre,post);
                        }
                }
            }
        }
    }
    
    protected void showProperties()
    {
        System.out.println("VoiceTrainer SRN Version " + VERSION_NUMBER);
        System.out.println("srnaToLoad        = " + srnaToLoad);
        System.out.println("srnaToSave        = " + srnaToSave);
        System.out.println("learningRate      = " + learningRate);
        System.out.println("momentumTerm      = " + momentumTerm);
        System.out.println("randomSeed        = " + randomSeed);
        System.out.println("predictionDelay   = " + predictionDelay);
        System.out.println("preserveFactor    = " + preserveFactor);
        System.out.println("numInputCopies    = " + numInputCopies);
        System.out.println("inputPatternWidth = " + inputPatternWidth);
        System.out.println("batchesPerTrial   = " + batchesPerTrial);
        System.out.println("hiddenPerUnit     = " + hiddenUnitsPerInputVector);
        System.out.println("restrictedFeedforward = " + restrictedFeedforward);
        System.out.println("trials            = " + trials);
        System.out.println("inputList         = " + inputList);
        System.out.println("probeList         = " + probeList);
        System.out.println("clearContextUnits = " + clearContextUnits);
        System.out.println("softClampingScalar = " + softClampingScalar);
        System.out.println("softClamping = " + softClamping);
        System.out.println("inputScaleFactor = " + inputScaleFactor);
        System.out.println("initialRandomWeightFactor = " + initialRandomWeightFactor);
        
    }
    
    void configureAveragerProperties(SimpleRecurrentNetAverager srna)
    {
        // set input neurons
        
        for (int i = 1; i < numInputCopies; i++)
        {
            for (       int neuron = i * inputPatternWidth;
            neuron < ((i+1) * inputPatternWidth);
            neuron++)
            {
                srna.setPreserveFactor(neuron, getPreserveFactorForCopy(i));      // first set of input units
                System.out.println("Neuron " + neuron + " preserve = " + getPreserveFactorForCopy(i));
            }
        }
        // Set hidden units?
        
        for (int i = 1; i < numInputCopies; i++)
        {
            int hiddenNeuronStart = (inputPatternWidth * numInputCopies) + i * hiddenUnitsPerInputVector;
            int hiddenNeuronCease = hiddenNeuronStart + hiddenUnitsPerInputVector;
            
            for (int neuron = hiddenNeuronStart; neuron < hiddenNeuronCease; neuron++)
            {
                srna.setPreserveFactor(neuron, getPreserveFactorForCopy(i));
                System.out.println("Hidden neuron " + neuron + " preserve = " + getPreserveFactorForCopy(i));
            }
        }
        
        
    }
    
    protected double getPreserveFactorForCopy(int copyNumber)
    {
        if (copyNumber == 0)
        {
            return 0;		// First input copy should preserve nothing, and always reflect the present
        }
        return Math.pow(preserveFactor, (double) copyNumber);
    }
    
    
    protected List readSoundPatterns(List filenames, boolean replicateFlag)
    {
        ArrayList result = new ArrayList();
        Iterator i = filenames.iterator();
        while (i.hasNext())
        {
            result.add(readSoundPattern((String)i.next(), replicateFlag));
        }
        return result;
        
    }
    
    protected List readSoundPattern(String filename, boolean replicateFlag)
    {
        try
        {
            ArrayList result = new ArrayList();
            BufferedReader f = new BufferedReader(new FileReader(filename));
            while (f.ready())
            {
                try
                {
                    String s = f.readLine();
                    StringTokenizer t = new StringTokenizer(s, " ", false);
                    double [] timestep = tokenizerToDoubles(t);
                    if (normalizeInputs) timestep = VectorComparer.normalize(timestep);
                    if (divideByMaxInputs) timestep = divideByMaxInputs(timestep);
                    for (int stutter = 0; stutter < stutterTimeSteps; stutter++)
                    {
                        if (replicateFlag)
                            result.add(replicateInputVector(timestep));
                        else
                            result.add(timestep);
                    }
                }
                catch (Exception e)
                {
                    throw new RuntimeException("Exception occured while processing file " + filename+":\n"+e.getMessage());
                }
            }
            return result;
        }
        catch (Exception e)
        {
            e.printStackTrace();
            throw new RuntimeException("Could not read sound pattern file "+filename+" for processing.");
        }
    }
    
    protected double [] divideByMaxInputs(double [] v)
    {
        double [] result = new double[v.length];
        double max = v[0];
        for (int i = 1; i < v.length; i++)
        {
            if (v[i] > max) max = v[i];
        }
        for (int i = 0; i < v.length; i++)
        {
            result[i] = v[i] / max;
        }
        return result;
    }
    
    protected double [] tokenizerToDoubles(StringTokenizer t)
    {
        if ((t.countTokens() != inputPatternWidth) && (t.countTokens() != outputPatternWidth))
        {
            throw new RuntimeException("Error: inputPatternWidth=" + inputPatternWidth + ", outputPatternWidth = "+ outputPatternWidth +" but found " +t.countTokens() + " values in an input vector.");
        }
        double [] result = new double[t.countTokens()];
        int i = 0;
        while (t.hasMoreTokens())
        {
            result[i] = new Double(t.nextToken()).doubleValue();
            i++;
        }
        return result;
    }
    
    protected List stringToList(String passed)
    {
        ArrayList result = new ArrayList();
        String s = new String(passed);
        if (!s.endsWith(" ")) s+=" ";
        StringTokenizer t = new StringTokenizer(s, " ", false);
        while (t.hasMoreTokens())
        {
            result.add(t.nextToken());
        }
        return result;
    }
}
